package io.privy.auth

import io.privy.auth.internal.InternalAuthState
import io.privy.auth.persistence.AuthSessionDataStore
import io.privy.auth.persistence.InternalAuthSessionTokens
import io.privy.di.KmpAppScope
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.firstOrNull
import me.tatarka.inject.annotations.Inject


public interface InternalAuthStateRepository: AuthStateRepository {
  public suspend fun updateInternalAuthState(internalAuthState: InternalAuthState)

  public suspend fun loadPersistedSessionTokens(): InternalAuthSessionTokens?
}

@Inject
@KmpAppScope
public class RealAuthStateRepository(
  private val authSessionDataStore: AuthSessionDataStore
): InternalAuthStateRepository {
  // Define auth state state flow with NotReady as initial value
  private val _internalAuthState: MutableStateFlow<InternalAuthState> = MutableStateFlow(InternalAuthState.NotReady)

  override val internalAuthState: StateFlow<InternalAuthState> = this._internalAuthState.asStateFlow()

  override suspend fun loadPersistedSessionTokens(): InternalAuthSessionTokens? {
    return authSessionDataStore.internalAuthSessionTokens.firstOrNull()
  }

  override suspend fun updateInternalAuthState(internalAuthState: InternalAuthState) {
    when(internalAuthState) {
      InternalAuthState.NotReady -> {}
      is InternalAuthState.Authenticated -> {
        authSessionDataStore.saveAuthSession(internalAuthState.session)
      }
      InternalAuthState.Unauthenticated -> {
        authSessionDataStore.clearAuthSession()
      }
    }

    _internalAuthState.value = internalAuthState
  }
}