package io.privy.auth.session

import kotlinx.serialization.ExperimentalSerializationApi
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.encodeToJsonElement
import kotlinx.serialization.modules.SerializersModule
import kotlinx.serialization.modules.polymorphic
import me.tatarka.inject.annotations.Inject

public interface AuthSessionResponseDeserializer {
  public fun deserialize(jsonResponse: String): AuthSessionResponse

  public fun deserializeUser(jsonResponse: String): RefreshUserResponse
}

@Inject
public class RealAuthSessionResponseDeserializer: AuthSessionResponseDeserializer {
  private val authSessionSerializer = Json {
    prettyPrint = true
    isLenient = true
    ignoreUnknownKeys = true
    explicitNulls = false

    serializersModule = SerializersModule {
      polymorphic(LinkedAccountResponse::class) {
        subclass(
          LinkedAccountResponse.WalletAccount::class,
          LinkedAccountResponse.WalletAccount.serializer()
        )
        subclass(
          LinkedAccountResponse.CustomAuth::class,
          LinkedAccountResponse.CustomAuth.serializer()
        )
        subclass(
          LinkedAccountResponse.PhoneNumberAccount::class,
          LinkedAccountResponse.PhoneNumberAccount.serializer()
        )
        subclass(
          LinkedAccountResponse.EmailAddressAccount::class,
          LinkedAccountResponse.EmailAddressAccount.serializer()
        )

        defaultDeserializer { LinkedAccountResponse.UnknownAccount.serializer() }
      }

      classDiscriminator = "type"
    }
  }

  override fun deserialize(jsonResponse: String): AuthSessionResponse {
    return authSessionSerializer.decodeFromString<AuthSessionResponse>(jsonResponse)
  }

  override fun deserializeUser(jsonResponse: String): RefreshUserResponse {
    return authSessionSerializer.decodeFromString<RefreshUserResponse>(jsonResponse)
  }
}