package io.privy.auth

import io.privy.auth.customAuth.TokenProvider
import io.privy.auth.internal.InternalAuthSession
import io.privy.auth.internal.InternalAuthState
import io.privy.auth.internal.LoginMethod
import io.privy.auth.internal.SessionUpdateAction
import io.privy.auth.jwt.DecodeJwt
import io.privy.auth.otp.OtpRequestType
import io.privy.auth.persistence.InternalAuthSessionTokens
import io.privy.logging.PrivyLogger
import io.privy.network.toResult
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.IO
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.launch
import me.tatarka.inject.annotations.Inject

@Inject
public class RealAuthManager(
    private val authRepository: AuthRepository,
    private val authStateRepository: InternalAuthStateRepository,
    private val privyLogger: PrivyLogger,
    private val decodeJwt: DecodeJwt,
    private val authRefreshService: AuthRefreshService,
    privyUserCreator: (AuthManager) -> PrivyUser
) : AuthManager {
  private val scope = CoroutineScope(SupervisorJob() + Dispatchers.IO)

  // Create PrivyUser for internal use, but ONLY expose it publicly if there is an authenticated
  // session
  override val privyUser: PrivyUser = privyUserCreator(this)

  // Custom auth
  private var tokenProvider: TokenProvider? = null

  override fun setTokenProvider(tokenProvider: TokenProvider) {
    this.tokenProvider = tokenProvider
  }

  init {
    this.restoreSession()
  }

  /**
   * When the SDK is initialized, auth state is set to "NotReady" Privy is not ready to use until
   * auth state gets updated to something other than "NotReady". Thus, restoring session MUST update
   * auth state to authenticated or unauthenticated
   */
  private fun restoreSession() {
    scope.launch {
      privyLogger.internal("Attempting to restore prior session.")

      // Check if prior auth session exists
      val priorSessionTokens = authStateRepository.loadPersistedSessionTokens()

      privyLogger.error("PRIOR SESSION TOKENS: $priorSessionTokens")

      if (priorSessionTokens != null) {
        privyLogger.internal("Prior session exists!")

        // as part of restoring, if tokens are available, we should refresh session to fetch user
        priorSessionTokens.refreshSession()
      } else {
        privyLogger.internal("No prior session, user is unauthenticated!")
        // no prior session - trigger logout flow which will clear all data and set user to
        // unauthenticated
        logout()
      }
    }
  }

  /**
   * Publicly exposed login method, which converts internal auth session to user for convenience.
   */
  override suspend fun login(loginType: LoginType): Result<PrivyUser> {
    return internalLogin(loginType).map {
      // on successful login, return privy user
      privyUser
    }
  }

  override suspend fun sendOtp(otpRequestType: OtpRequestType): Result<Unit> {
    val result = authRepository.sendOtp(otpRequestType).toResult()

    return result.fold(
        onSuccess = { success ->
          if (success) {
            Result.success(Unit)
          } else {
            Result.failure(AuthenticationException("Error sending OTP."))
          }
        },
        onFailure = {
          // pass back the failure
          Result.failure(it)
        })
  }

  private suspend fun internalLogin(loginType: LoginType): Result<InternalAuthSession> {
    val authResult =
        authRepository.authenticate(loginType = loginType).toResult().onFailure {
          privyLogger.internal("Login failed, error: $it")
        }

    return handleAuthResponse(authResult = authResult)
  }

  // Helper method that takes all required actions when we get a new auth response
  private suspend fun handleAuthResponse(
      authResult: Result<InternalAuthSession>
  ): Result<InternalAuthSession> {
    authResult.fold(
        onSuccess = { internalAuthSession ->
          when (internalAuthSession.sessionUpdateAction) {
            SessionUpdateAction.Set -> updateAuthSession(internalAuthSession)
            // If we receive a "clear" update action, log the user out
            SessionUpdateAction.Clear -> this.logout()
            // Ignore case is odd, because there are times we conditionally ignore. for now, just
            // update
            SessionUpdateAction.Ignore -> updateAuthSession(internalAuthSession)
          }
        },
        onFailure = {
          // update auth state to unauthenticated, then return failure

          // if (it is PrivyApiException) {
          // in the future, convert this into a public facing exception
          // }

          // TODO: maybe want to create a way to transform throwable to privy error?
          this.logout()
        })

    return authResult
  }

  public override fun <T> authenticatedOrDefault(
      onAuthenticated: (InternalAuthSession) -> T,
      default: () -> T
  ): T {
    // Current session
    val currAuthState = authStateRepository.internalAuthState.value

    return if (currAuthState is InternalAuthState.Authenticated) {
      onAuthenticated(currAuthState.session)
    } else {
      default()
    }
  }

  // Helper function to ensure user is authenticated before making wallet call
  // If authenticated: returns result.success with associated session
  // If unauthenticated: returns result.failure
  public override suspend fun <T> ensureAuthenticated(
      onAuthenticated: suspend (InternalAuthSession) -> Result<T>
  ): Result<T> {
    return refreshSessionIfNeeded()
        .fold(onSuccess = { onAuthenticated(it) }, onFailure = { Result.failure(it) })
  }

  override suspend fun refreshUser(): Result<InternalAuthSession> {
    // Current session
    val currAuthState = authStateRepository.internalAuthState.value

    return if (currAuthState is InternalAuthState.Authenticated) {
      currAuthState.session.refreshUser()
    } else {
      Result.failure(
          AuthenticationException("User must be authenticated before calling refresh user."))
    }
  }

  override suspend fun refreshSession(): Result<InternalAuthSession> {
    // Current session
    val currAuthState = authStateRepository.internalAuthState.value

    return if (currAuthState is InternalAuthState.Authenticated) {
      currAuthState.session.refreshSession()
    } else {
      Result.failure(AuthenticationException("User must be authenticated before calling refresh."))
    }
  }

  override suspend fun logout() {
    authStateRepository.updateInternalAuthState(
        internalAuthState = InternalAuthState.Unauthenticated)
  }

  override suspend fun refreshSessionIfNeeded(): Result<InternalAuthSession> {
    // Current session
    val currAuthState = authStateRepository.internalAuthState.value

    return if (currAuthState is InternalAuthState.Authenticated) {
      currAuthState.session.refreshSessionIfNeeded()
    } else {
      Result.failure(
          AuthenticationException("User must be authenticated before calling refresh session."))
    }
  }

  private suspend fun updateAuthSession(authSession: InternalAuthSession) {
    // user is authenticated - update auth state
    // Note: external auth state should automatically update after setting internal state.
    authStateRepository.updateInternalAuthState(
        internalAuthState = InternalAuthState.Authenticated(session = authSession))
  }

  private suspend fun InternalAuthSession.refreshSessionIfNeeded(): Result<InternalAuthSession> {
    val jwt = decodeJwt(this.accessToken)

    return if (jwt.isExpired()) {
      // If session is expired, try refreshing
      this.refreshSession()
    } else {
      // If session is already valid, return itself
      Result.success(this)
    }
  }

  // Helper function to refresh user in the scope of an existing session
  private suspend fun InternalAuthSession.refreshUser(): Result<InternalAuthSession> {
    val jwt = decodeJwt(this.accessToken)

    return if (jwt.isExpired()) {
      // If session is expired, refresh it, which will also refresh user
      this.refreshSession()
    } else {
      // If session is already valid, just refresh the user
      return authRepository
          .refreshUser(this.accessToken)
          .toResult()
          .fold(
              onSuccess = { refreshedUser ->
                val updatedSession = this.copy(user = refreshedUser)
                updateAuthSession(updatedSession)
                Result.success(updatedSession)
              },
              onFailure = { Result.failure(it) })
    }
  }

  // Helper function to refresh user in the scope of an existing session
  private suspend fun InternalAuthSession.refreshSession(): Result<InternalAuthSession> {
    return when (this.loginMethod) {
      LoginMethod.CustomAccessToken -> reAuthenticateWithCustomAuth()
      else -> {
        return refreshTokensWithPrivy(
            currentAccessToken = this.accessToken,
            currentRefreshToken = this.refreshToken,
            currentLoginMethod = this.loginMethod)
      }
    }
  }

  private suspend fun InternalAuthSessionTokens.refreshSession(): Result<InternalAuthSession> {
    return when (this.loginMethod) {
      LoginMethod.CustomAccessToken -> reAuthenticateWithCustomAuth()
      else -> {
        return refreshTokensWithPrivy(
            currentAccessToken = this.accessToken,
            currentRefreshToken = this.refreshToken,
            currentLoginMethod = this.loginMethod)
      }
    }
  }

  private suspend fun refreshTokensWithPrivy(
      currentAccessToken: String,
      currentRefreshToken: String,
      currentLoginMethod: LoginMethod,
  ): Result<InternalAuthSession> {
    // Using auth refresh service guarantees thread safety
    val refreshResult =
        authRefreshService
            .refreshSession(
                authToken = currentAccessToken,
                refreshToken = currentRefreshToken,
                currentLoginMethod = currentLoginMethod)
            .toResult()

    return handleAuthResponse(authResult = refreshResult)
  }

  private suspend fun reAuthenticateWithCustomAuth(): Result<InternalAuthSession> {
    val customAuthAccessToken = tokenProvider?.invoke()

    if (customAuthAccessToken == null) {
      // since reauthenticating user failed, log them out
      privyLogger.debug("Custom access token from tokenProvider null. Logging user out.")
      this.logout()
      return Result.failure(
          AuthenticationException("Custom access token from tokenProvider null. Logging user out."))
    }

    return this.internalLogin(
        loginType = LoginType.CustomAccessToken(token = customAuthAccessToken))
  }
}
