package io.privy.auth.session.internal

import io.privy.auth.internal.InternalLinkedAccount
import io.privy.auth.internal.InternalPrivyUser
import io.privy.auth.session.LinkedAccountResponse
import io.privy.auth.session.PrivyUserResponse

internal fun PrivyUserResponse.toInternalPrivyUser(): InternalPrivyUser {
  return with(this) {
    InternalPrivyUser(
        id = id, linkedAccounts = linkedAccounts.mapNotNull { it.toInternalLinkedAccount() })
  }
}

private fun LinkedAccountResponse.toInternalLinkedAccount(): InternalLinkedAccount? {
  return when (this) {
    is LinkedAccountResponse.CustomAuth ->
        InternalLinkedAccount.CustomAuth(
            customUserId = customUserId,
            firstVerifiedAt = firstVerifiedAt,
            latestVerifiedAt = latestVerifiedAt,
        )
    is LinkedAccountResponse.WalletAccount ->
        if (this.isPrivyWallet()) {
          this.toInternalEmbeddedWalletOrNull()
        } else {
          // FOR NOW, we don't support external wallets, but when we do, add parsing logic here
          null
        }
    is LinkedAccountResponse.PhoneNumberAccount ->
        InternalLinkedAccount.PhoneAccount(phoneNumber = phoneNumber)
    is LinkedAccountResponse.EmailAddressAccount ->
        InternalLinkedAccount.EmailAccount(emailAddress = emailAddress)
    is LinkedAccountResponse.UnknownAccount -> null
  }
}

private fun LinkedAccountResponse.WalletAccount.toInternalEmbeddedWalletOrNull():
    InternalLinkedAccount.EmbeddedWalletAccount? {
  if (hdWalletIndex == null) {
    // privy embedded wallet should always have hd wallet index
    return null
  }

  return if (chainType == "solana") {
    InternalLinkedAccount.EmbeddedSolanaWalletAccount(
        address = address,
        chainId = chainId,
        recoveryMethod = recoveryMethod,
        hdWalletIndex = hdWalletIndex,
    )
  } else {
    InternalLinkedAccount.EmbeddedEthereumWalletAccount(
        address = address,
        chainId = chainId,
        recoveryMethod = recoveryMethod,
        hdWalletIndex = hdWalletIndex,
        firstVerifiedAt = firstVerifiedAt,
        latestVerifiedAt = latestVerifiedAt)
  }
}
