package io.privy.auth

import io.privy.auth.internal.InternalLinkedAccount
import io.privy.auth.internal.embeddedEthereumWalletAccounts
import io.privy.auth.internal.embeddedSolanaWalletAccounts
import io.privy.auth.internal.primaryWalletForEntropyInfo
import io.privy.auth.internal.toEmbeddedWalletDetails
import io.privy.logging.PrivyLogger
import io.privy.wallet.EmbeddedWallet
import io.privy.wallet.EmbeddedWalletManager
import io.privy.wallet.ethereum.EmbeddedEthereumWallet
import io.privy.wallet.ethereum.EmbeddedEthereumWalletFactory
import io.privy.wallet.solana.EmbeddedSolanaWallet
import io.privy.wallet.solana.EmbeddedSolanaWalletFactory
import me.tatarka.inject.annotations.Assisted
import me.tatarka.inject.annotations.Inject

/**
 * RealPrivyUser is simply a public facing abstraction of a user's auth session. This class should
 * be stateless and should dynamically derive values as they are requested. This ensures the return
 * values will be up to date. i.e. when user.linkedAccounts is accessed, it will dynamically
 * determine the users' linked accounts based on the latest auth session value
 */
@Inject
public class RealPrivyUser(
    @Assisted private val authManager: AuthManager,
    private val embeddedEthereumWalletFactory: EmbeddedEthereumWalletFactory,
    private val embeddedSolanaWalletFactory: EmbeddedSolanaWalletFactory,
    private val embeddedWalletManager: EmbeddedWalletManager,
    private val privyLogger: PrivyLogger,
) : PrivyUser {
  // All properties on Privy user are derived, meaning they are determined at the time the value
  // is retrieved. This ensures that any updates to the internal session are taken into account and
  // all updates are reflected on the user object.
  override val id: String
    get() {
      return authManager.authenticatedOrDefault(
          onAuthenticated = { authSession -> authSession.user.id },
          default = {
            // If user is unauthenticated, return empty string for ID
            // This would only happen if app dev captured the user object when authenticated, and
            // then
            // user was logged out
            ""
          })
    }

  // Public facing LinkedAccounts
  override val linkedAccounts: List<LinkedAccount>
    get() {
      return internalLinkedAccounts.map { it.toLinkedAccount() }
    }

  // The identity token for this user, if configured in the Privy dashboard.
  override val identityToken: String?
    get() = authManager.authenticatedOrDefault({ it.identityToken }, { null })

  private val internalLinkedAccounts: List<InternalLinkedAccount>
    get() {
      return authManager.authenticatedOrDefault(
          onAuthenticated = { authSession -> authSession.user.linkedAccounts },
          default = {
            // If user us unauthenticated, return empty list
            // This would only happen if app dev captured the user object when authenticated, and
            // then
            // user was logged out
            emptyList()
          })
    }

  override val embeddedEthereumWallets: List<EmbeddedEthereumWallet>
    get() {
      val primaryEmbeddedWalletAccount = internalLinkedAccounts.primaryWalletForEntropyInfo()

      val embeddedWallets =
          primaryEmbeddedWalletAccount?.let { primaryWalletAccount ->
            internalLinkedAccounts.embeddedEthereumWalletAccounts().map {
              embeddedEthereumWalletFactory.create(
                  embeddedWalletDetails =
                      it.toEmbeddedWalletDetails(
                          primaryEmbeddedWallet = primaryEmbeddedWalletAccount))
            }
          }

      return embeddedWallets ?: emptyList()
    }

  override val embeddedSolanaWallets: List<EmbeddedSolanaWallet>
    get() {
      val primaryEmbeddedWalletAccount = internalLinkedAccounts.primaryWalletForEntropyInfo()

      val embeddedWallets =
          primaryEmbeddedWalletAccount?.let { primaryWalletAccount ->
            internalLinkedAccounts.embeddedSolanaWalletAccounts().map {
              embeddedSolanaWalletFactory.create(
                  embeddedWalletDetails =
                      it.toEmbeddedWalletDetails(
                          primaryEmbeddedWallet = primaryEmbeddedWalletAccount))
            }
          }

      return embeddedWallets ?: emptyList()
    }

  override suspend fun refresh(): Result<Unit> {
    return authManager.refreshUser().map {
      // refresh user returns InternalAuthSession, which we don't want to expose to user, so map
      // it to Unit. This way, developer can determine if refresh was successful
      Unit
    }
  }

  override suspend fun getAccessToken(): Result<String> {
    return authManager.refreshSessionIfNeeded().map { it.accessToken }
  }

  override suspend fun createEthereumWallet(
      allowAdditional: Boolean
  ): Result<EmbeddedEthereumWallet> {
    return authManager.ensureAuthenticated { authSession ->
      val accessToken = authSession.accessToken

      val primaryEmbeddedWalletAccount = linkedAccounts.primaryEmbeddedEthereumWalletAccountOrNull()

      if (primaryEmbeddedWalletAccount == null) {
        // User does not have an embedded wallet yet, need to create primary wallet
        createPrimaryEthereumWallet(accessToken = accessToken)
      } else {
        // User already has a primary wallet, create additional if allowed
        createAdditionalWalletIfAllowed(
            allowAdditional = allowAdditional,
            accessToken = accessToken,
            primaryWalletAddress = primaryEmbeddedWalletAccount.address)
      }
    }
  }

  override suspend fun createSolanaWallet(): Result<EmbeddedSolanaWallet> {
    return authManager.ensureAuthenticated { authSession ->
      val accessToken = authSession.accessToken

      val primarySolanaWalletAccount = linkedAccounts.primaryEmbeddedSolanaWalletAccountOrNull()

      if (primarySolanaWalletAccount == null) {
        // User does not have a Solana embedded wallet yet, need to create primary wallet
        createPrimarySolanaWallet(accessToken = accessToken)
      } else {
        // User already has a primary wallet
        privyLogger.error("User already has a solana wallet.")

        Result.failure(
            exception = EmbeddedWalletException(message = "User already has a solana wallet."))
      }
    }
  }

  /**
   * Internal method in charge of creating the user's / first Ethereum primary wallet (hdIndex == 0)
   * Caller of this method should verify user is authenticated and pass in a valid access token
   */
  private suspend fun createPrimaryEthereumWallet(
      accessToken: String
  ): Result<EmbeddedEthereumWallet> {
    return embeddedWalletManager
        .createEthereumWallet(
            accessToken = accessToken,
            existingSolanaWalletAddress =
                linkedAccounts.primaryEmbeddedSolanaWalletAccountOrNull()?.address)
        .flatMap { newWalletAddress ->
          onWalletCreated(
              newWalletAddress = newWalletAddress, getWallets = { embeddedEthereumWallets })
        }
  }

  /**
   * Internal method in charge of creating the user's / first Solana primary wallet (hdIndex == 0)
   * Caller of this method should verify user is authenticated and pass in a valid access token
   */
  private suspend fun createPrimarySolanaWallet(accessToken: String): Result<EmbeddedSolanaWallet> {
    return embeddedWalletManager
        .createSolanaWallet(
            accessToken = accessToken,
            existingEthereumWalletAddress =
                linkedAccounts.primaryEmbeddedEthereumWalletAccountOrNull()?.address)
        .flatMap { newWalletAddress ->
          onWalletCreated(
              newWalletAddress = newWalletAddress, getWallets = { embeddedSolanaWallets })
        }
  }

  /**
   * Internal method in charge of creating an additional wallet for the user's if dev specifies
   * additional wallets are allowed. Caller of this method should verify user is authenticated and
   * pass in a valid access token, and should verify a user already has a primary embedded wallet
   * account.
   */
  private suspend fun createAdditionalWalletIfAllowed(
      allowAdditional: Boolean,
      accessToken: String,
      primaryWalletAddress: String,
  ): Result<EmbeddedEthereumWallet> {
    return if (allowAdditional) {
      createAdditionalWallet(accessToken = accessToken, primaryWalletAddress = primaryWalletAddress)
    } else {
      // Dev didn't specify allowAdditional == true, so throw an error
      val errorMessage =
          "User already has an embedded wallet. To create an additional wallet, set allowAdditional to true."
      privyLogger.error(errorMessage)

      Result.failure(exception = EmbeddedWalletException(message = errorMessage))
    }
  }

  /**
   * Internal method in charge of creating an additional wallet for the user's (hdIndex > 0) Caller
   * of this method should verify user is authenticated and pass in a valid access token, and should
   * verify a user already has a primary embedded wallet account.
   */
  private suspend fun createAdditionalWallet(
      accessToken: String,
      primaryWalletAddress: String,
  ): Result<EmbeddedEthereumWallet> {
    // TODO: SPECIFY CHAIN TYPE WHEN GETTING "embeddedWalletAccounts"
    val currentWalletCount = internalLinkedAccounts.embeddedEthereumWalletAccounts().count()

    // There's actually no limit to how many wallets a user can have, but adding this check
    // client side as a sanity check, ensuring someone isn't creating 100s of wallets
    if (currentWalletCount >= maxWalletsAllowed) {
      return Result.failure(
          exception =
              EmbeddedWalletException(
                  message =
                      "User already has maximum amount of wallets ($maxWalletsAllowed) allowed."))
    }

    // The new wallets HD index should be the current highest wallet's index + 1
    // which is the same as the total count of embedded wallets
    return embeddedWalletManager
        .createAdditionalWallet(
            accessToken = accessToken,
            primaryWalletAddress = primaryWalletAddress,
            hdWalletIndex = currentWalletCount)
        .flatMap { newWalletAddress ->
          onWalletCreated(
              newWalletAddress = newWalletAddress, getWallets = { embeddedEthereumWallets })
        }
  }

  /**
   * A chain agnostic, reusable function that force refreshes auth session when new wallet is
   * created, which will retrieve the latest linked accounts, including the newly created wallet.
   *
   * @param newWalletAddress The address of the newly created wallet
   * @param getWallets A lambda that allows the caller to specify the list of wallets to search
   *   from. This allows the caller to specify the EmbeddedWallet type
   */
  private suspend fun <T : EmbeddedWallet> onWalletCreated(
      newWalletAddress: String,
      getWallets: () -> List<T>
  ): Result<T> {
    privyLogger.info("New embedded wallet created. Wallet address: $newWalletAddress")

    return authManager.refreshSession().flatMap {
      // Get newly created embedded wallet from list
      // important that we call getWallets() after refreshing the session
      val newEmbeddedWallet = getWallets().firstOrNull { it.address == newWalletAddress }

      return if (newEmbeddedWallet == null) {
        // Shouldn't happen
        Result.failure(
            exception = EmbeddedWalletException(message = "Failed to create embedded wallet."))
      } else {
        // Once session is refreshed, try connecting wallet
        embeddedWalletManager.attemptConnectWalletInBackground()

        // return embedded wallet
        Result.success(newEmbeddedWallet)
      }
    }
  }

  override fun toString(): String {
    val defaultToString = super.toString()

    return """ 
    PrivyUser(
      ID: '$id', 
      
      Linked Accounts:
      ${linkedAccounts.map { it.prettyPrint() }}, 
      
      Instance: ${defaultToString}, 
      
      Identity Token: ${identityToken?.take(10)?.let { "$it..." } ?: "None"}
    )
    """
        .trimIndent()
  }

  private fun EmbeddedWallet.prettyPrint(): String {
    return """
    EmbeddedWallet(
      address=$address,
      chainType=$chainType,
      hdWalletIndex=$hdWalletIndex,
      chainId=$chainId,
      recoveryMethod=$recoveryMethod,
    )
  """
        .trimIndent()
  }

  private fun LinkedAccount.prettyPrint(): String {
    return when (this) {
      is LinkedAccount.CustomAuth ->
          """
        
        CustomAuth(
          User id: ${this.customUserId} 
        )
         
      """
              .trimIndent()
      is LinkedAccount.GoogleOAuthAccount ->
          """
        
        GoogleOAuth(
          Subject: ${this.subject}
          Email: ${this.email}
          Name: ${this.name ?: "N/A"}
          First Verified At: ${this.firstVerifiedAt}
          Latest Verified At: ${this.latestVerifiedAt}
        )
         
      """
              .trimIndent()
      is LinkedAccount.TwitterOAuthAccount ->
          """
        
        TwitterOAuth(
          Subject: ${this.subject}
          Username: ${this.username}
          Name: ${this.name ?: "N/A"}
          Email: ${this.email ?: "N/A"}
          Profile Picture URL: ${this.profilePictureUrl ?: "N/A"}
          First Verified At: ${this.firstVerifiedAt}
          Latest Verified At: ${this.latestVerifiedAt}
        )
         
      """
              .trimIndent()
      is LinkedAccount.DiscordOAuthAccount ->
          """
        
        DiscordOAuth(
          Subject: ${this.subject}
          Username: ${this.username}
          Email: ${this.email ?: "N/A"}
          First Verified At: ${this.firstVerifiedAt}
          Latest Verified At: ${this.latestVerifiedAt}
        )
         
      """
              .trimIndent()
      is LinkedAccount.EmailAccount ->
          """
        
        Email(
          address: ${this.emailAddress} 
        )
         
      """
              .trimIndent()
      is LinkedAccount.EmbeddedEthereumWalletAccount ->
          """
        
        EthereumWallet(
          address=$address,
          hdWalletIndex=$hdWalletIndex,
          chainId=$chainId,
          recoveryMethod=$recoveryMethod,
        )
        
      """
              .trimIndent()
      is LinkedAccount.EmbeddedSolanaWalletAccount ->
          """
        
        SolanaWallet(
          address=$address,
          hdWalletIndex=$hdWalletIndex,
          chainId=$chainId,
          recoveryMethod=$recoveryMethod,
        )
        
      """
              .trimIndent()
      is LinkedAccount.ExternalWalletAccount ->
          """
        
        ExternalWallet(
          address=$address,
          chainId=$chainId,
          chainType=${chainType.name},
          recoveryMethod=$connectorType
        )
        
      """
              .trimIndent()
      is LinkedAccount.PhoneAccount ->
          """
        
        SMS(
          phoneNumber: ${this.phoneNumber},
        )
         
      """
              .trimIndent()
    }
  }

  override fun equals(other: Any?): Boolean {
    if (other !is RealPrivyUser) {
      return false
    }

    // A user is equal to another user if they have the same ID and same set of
    // linked accounts.
    return id == other.id && linkedAccounts == other.linkedAccounts
  }

  override fun hashCode(): Int {
    var result = id.hashCode()
    result = 31 * result + linkedAccounts.hashCode()
    return result
  }

  private companion object {
    const val maxWalletsAllowed = 10
  }
}
