package io.privy.auth.oAuth

import io.ktor.http.Url
import io.privy.auth.AuthManager
import io.privy.auth.AuthenticationException
import io.privy.auth.LoginType
import io.privy.auth.PrivyUser
import io.privy.logging.PrivyLogger
import kotlin.Result
import me.tatarka.inject.annotations.Inject

/**
 * Implementation of the LoginWithOAuth interface that delegates to an OAuthHandler. This class
 * serves as a bridge between the public API and the internal OAuth handling mechanism, providing a
 * clean separation of concerns in the authentication system.
 */
@Inject
public class RealLoginWithOAuth(
    private val oAuthHandler: OAuthHandler,
    private val pkceHelper: PKCEHelper,
    private val authManager: AuthManager,
    private val logger: PrivyLogger,
) : LoginWithOAuth {

  /**
   * Initiates the OAuth login flow by delegating to the OAuthHandler. This method simply forwards
   * the request to the underlying OAuthHandler implementation, which handles the actual OAuth
   * authentication process.
   *
   * @param oAuthProvider The OAuth provider to authenticate with
   * @param appUrlScheme The URL scheme for handling redirects
   * @return Result containing the authenticated PrivyUser on success, or an exception on failure
   */
  override suspend fun login(
      oAuthProvider: OAuthProvider,
      appUrlScheme: String
  ): Result<PrivyUser> {

    // Validate that app URL scheme is provided, as it's required for the redirect
    if (appUrlScheme.isEmpty()) {
      logger.error("OAuth login failed: App URL scheme not provided")
      return Result.failure(AuthenticationException("App URL scheme must be provided"))
    }


    // Generate PKCE (Proof Key for Code Exchange) parameters for secure authorization
    val codeVerifier = pkceHelper.generateCodeVerifier()
    val codeChallenge = pkceHelper.deriveCodeChallenge(codeVerifier)
    val stateCode = pkceHelper.generateStateCode()

    // Prepare and launch the OAuth URL
    val launchResult =
        prepareAndLaunchInternal(oAuthProvider, appUrlScheme, codeChallenge, stateCode)

    if (launchResult.isFailure) {
      logger.error("Failed to launch OAuth browser")
      return Result.failure(AuthenticationException("Failed to launch OAuth browser"))
    }

    // Wait for the custom tab result
    val customTabResult = OAuthDataManager.awaitCustomTabResult()

    // Handle the custom tab result
    val intentDataString =
        customTabResult.getOrElse {
          return Result.failure(it)
        }

    // Extract and validate OAuth redirect parameters
    val oAuthRedirectQueryParams = extractOAuthRedirectParamsOrNull(intentDataString)
    if (oAuthRedirectQueryParams == null) {
      logger.error("OAuth redirect missing required parameters")
      return Result.failure(
          AuthenticationException("Invalid OAuth redirect - missing required parameters"))
    }

    // Validate that the received state matches the expected state to prevent CSRF attacks
    if (oAuthRedirectQueryParams.state != stateCode) {
      logger.error("OAuth state mismatch")
      return Result.failure(AuthenticationException("OAuth state mismatch"))
    }

    logger.debug("OAuth redirect validated, completing authentication")
    return authManager.login(
        LoginType.OAuth(
            provider = oAuthProvider,
            authorizationCode = oAuthRedirectQueryParams.code,
            codeVerifier = codeVerifier,
            stateCode = oAuthRedirectQueryParams.state))
  }

  /**
   * Prepares and launches the OAuth authentication flow. Generates the OAuth URL and opens it in a
   * browser tab.
   *
   * @param oAuthProvider The OAuth provider to authenticate with
   * @param appUrlScheme The URL scheme for handling the redirect
   * @param codeChallenge The PKCE code challenge
   * @param stateCode The state code for verifying the response
   * @return Result indicating success or failure of launching the authentication flow
   */
  private suspend fun prepareAndLaunchInternal(
      oAuthProvider: OAuthProvider,
      appUrlScheme: String,
      codeChallenge: String,
      stateCode: String
  ): Result<Unit> {
    return authManager
        .generateOAuthUrl(
            oAuthInitRequest =
                OAuthInitRequest(
                    provider = oAuthProvider,
                    redirectUri =
                        if (appUrlScheme.contains("://")) appUrlScheme else "$appUrlScheme://",
                    codeChallenge = codeChallenge,
                    stateCode = stateCode))
        .fold(
            onSuccess = { initData ->
              oAuthHandler.launchCustomTab(initData.url)
              Result.success(Unit)
            },
            onFailure = { error -> Result.failure(error) })
  }

  /**
   * Extracts OAuth redirect parameters from a URL or returns null if the URL is not a valid OAuth
   * redirect.
   */
  private fun extractOAuthRedirectParamsOrNull(url: String): OAuthRedirectQueryParams? {
    if (url.isEmpty()) return null

    return try {
      val parsedUrl = Url(url)
      val params = parsedUrl.parameters

      val code = params["privy_oauth_code"]
      val state = params["privy_oauth_state"]

      if (code == null || state == null) return null

      OAuthRedirectQueryParams(code, state)
    } catch (_: Exception) {
      null
    }
  }
}
