package io.privy.auth

import io.ktor.resources.Resource
import io.privy.auth.LoginType.CustomAccessToken
import io.privy.auth.internal.InternalAuthSession
import io.privy.auth.internal.LoginMethod
import io.privy.auth.otp.OtpRequestType
import io.privy.auth.session.AuthSessionResponse
import io.privy.auth.session.AuthSessionResponseDeserializer
import io.privy.auth.session.internal.MapAuthSessionResponseToInternalAuthSession
import io.privy.logging.PrivyLogger
import io.privy.network.ApiResult
import io.privy.network.AuthType
import io.privy.network.KtorWrapper
import io.privy.network.map
import kotlinx.serialization.Serializable
import me.tatarka.inject.annotations.Inject

@Inject
public class RealAuthRepository(
  private val ktorWrapper: KtorWrapper,
  private val privyLogger: PrivyLogger,
  private val authSessionResponseDeserializer: AuthSessionResponseDeserializer,
  private val mapAuthSessionResponseToInternalAuthSession: MapAuthSessionResponseToInternalAuthSession,
): AuthRepository {
  override suspend fun authenticate(loginType: LoginType): ApiResult<InternalAuthSession> {
    val authenticateRequest = AuthenticateResource(
      loginTypeApiPath = loginType.apiPath()
    )

    val authenticateResponse: ApiResult<String> = ktorWrapper.postResult(
      resource = authenticateRequest,
      body = loginType.requestBody(),
      authType = AuthType.None,
    )

    return convertAuthResponseToInternalSession(
      authResponse = authenticateResponse,
      loginMethod = loginType.toLoginMethod(),
    )
  }

  override suspend fun refreshSession(
    authToken: String,
    refreshToken: String,
    currentLoginMethod: LoginMethod,
  ): ApiResult<InternalAuthSession> {
    val refreshResponse: ApiResult<String> = ktorWrapper.postResult(
      resource = RefreshResource,
      body = mapOf("refresh_token" to refreshToken),
      authType = AuthType.Authorization(accessToken = authToken),
    )

    return convertAuthResponseToInternalSession(
      authResponse = refreshResponse,
      loginMethod = currentLoginMethod,
    )
  }

  private fun convertAuthResponseToInternalSession(
    authResponse: ApiResult<String>,
    loginMethod: LoginMethod
  ): ApiResult<InternalAuthSession> {
    val mappedResponse = authResponse.map { jsonResponse ->
      authSessionResponseDeserializer.deserialize(jsonResponse = jsonResponse)
    }

    privyLogger.internal("Authenticate API response: $mappedResponse")

    val internalAuthSession = mappedResponse.map {
      mapAuthSessionResponseToInternalAuthSession.map(
        authSessionResponse = it,
        loginMethod = loginMethod,
      )
    }

    return internalAuthSession
  }

  override suspend fun sendOtp(otpRequestType: OtpRequestType): ApiResult<Boolean> {
    val sendOtpRequest = SendOtpResource(otpTypeApiPath = otpRequestType.apiPath())

    val sendOtpResponse: ApiResult<OtpInitResponse> = ktorWrapper.postResult(
      resource = sendOtpRequest,
      body = otpRequestType.requestBody(),
      authType = AuthType.None,
    )

    return sendOtpResponse.map { it.success }
  }
}

private fun LoginType.toLoginMethod(): LoginMethod {
  return when(this) {
    is LoginType.CustomAccessToken -> LoginMethod.CustomAccessToken
    is LoginType.Sms -> LoginMethod.Sms
    is LoginType.Email -> LoginMethod.Email
  }
}

@Resource("{otpTypeApiPath}/init")
private class SendOtpResource(
  val otpTypeApiPath: String,
)


@Resource("{loginTypeApiPath}/authenticate")
private class AuthenticateResource(
  val loginTypeApiPath: String,
)

@Resource("sessions")
private data object RefreshResource

private fun LoginType.apiPath(): String {
  return when(this) {
    is CustomAccessToken -> "custom_jwt_account"
    is LoginType.Sms -> "passwordless_sms"
    is LoginType.Email -> "passwordless"
  }
}

private fun LoginType.requestBody(): Map<String, String> {
  return when(this) {
    is CustomAccessToken -> mapOf("token" to this.token)
    is LoginType.Sms -> mapOf(
      "phoneNumber" to this.phoneNumber,
      "code" to this.code
    )
    is LoginType.Email -> mapOf(
      "email" to this.emailAddress,
      "code" to this.code
    )
  }
}

private fun OtpRequestType.apiPath(): String {
  return when(this) {
    is OtpRequestType.Email -> "passwordless"
    is OtpRequestType.Phone -> "passwordless_sms"
  }
}

private fun OtpRequestType.requestBody(): Map<String, String> {
  return when(this) {
    is OtpRequestType.Email -> mapOf("email" to this.email)
    is OtpRequestType.Phone -> mapOf("phoneNumber" to this.phone)
  }
}

@Serializable
private data class OtpInitResponse(val success: Boolean)