package io.privy.auth.persistence

import androidx.datastore.core.DataStore
import androidx.datastore.core.handlers.ReplaceFileCorruptionHandler
import androidx.datastore.preferences.core.PreferenceDataStoreFactory
import androidx.datastore.preferences.core.Preferences
import androidx.datastore.preferences.core.edit
import androidx.datastore.preferences.core.stringPreferencesKey
import io.privy.auth.internal.InternalAuthSession
import io.privy.auth.session.internal.InternalAuthSessionSerializer
import io.privy.di.KmpAppScope
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.map
import me.tatarka.inject.annotations.Inject
import okio.Path.Companion.toPath

public interface AuthSessionDataStore {
  public val internalAuthSession: Flow<InternalAuthSession?>

  public suspend fun saveAuthSession(session: InternalAuthSession)

  public suspend fun clearAuthSession()
}

@Inject
@KmpAppScope
public class RealAuthSessionDataStore(
  authSessionDataStoreHolder: AuthSessionDataStoreHolder,
  private val authSessionSerializer: InternalAuthSessionSerializer,
): AuthSessionDataStore {
  private val dataStore = authSessionDataStoreHolder.dataStore
  private val authSessionInfo = stringPreferencesKey(name = "auth_session_info")

  override val internalAuthSession: Flow<InternalAuthSession?> =
    authSessionDataStoreHolder.dataStore.data
      .map {
        // Grab json string from preferences
        preferences -> preferences[authSessionInfo]
      }
      .map { authSessionInfoString ->
        authSessionInfoString?.let { authSessionJsonString ->
          // If json string exists, attempt to deserialize it
          authSessionSerializer.deserializeOrNull(authSessionJsonString)
        }
      }

  override suspend fun saveAuthSession(session: InternalAuthSession) {
    dataStore.edit { prefs ->
      // Serialize session to json string, then store it in preferences
      prefs[authSessionInfo] = authSessionSerializer.serialize(session)
    }
  }

  override suspend fun clearAuthSession() {
    dataStore.edit { prefs ->
      // Clear out auth session
      prefs.remove(authSessionInfo)
    }
  }
}

// Preferences file must have .preferences_pb extension
internal const val dataStoreFileName = "auth-session.preferences_pb"

// For now, using Preferences data store and serializing data is fine.
// If our auth session object grows to something incredibly complex, we can
// migrate to protobuf data store
internal fun createDataStoreWithPath(
  corruptionHandler: ReplaceFileCorruptionHandler<Preferences>,
  producePath: () -> String
): DataStore<Preferences> =
  PreferenceDataStoreFactory.createWithPath(
    corruptionHandler = corruptionHandler,
    produceFile = { producePath().toPath() }
  )
