package io.privy.auth

import io.privy.auth.internal.InternalAuthSession
import io.privy.auth.internal.LoginMethod
import io.privy.auth.internal.SessionUpdateAction
import io.privy.di.KmpAppScope
import io.privy.network.ApiResult
import io.privy.network.map
import kotlinx.coroutines.Deferred
import kotlinx.coroutines.async
import kotlinx.coroutines.coroutineScope
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import me.tatarka.inject.annotations.Inject

/**
 * Thread safe mechanism to refresh session. This is required to prevent multiple calls to refresh
 * from the same session
 */
public interface AuthRefreshService {
  public suspend fun refreshSession(
      authToken: String,
      refreshToken: String,
      currentLoginMethod: LoginMethod
  ): ApiResult<InternalAuthSession>
}

@Inject
@KmpAppScope
public class RealAuthRefreshService(private val authRepository: AuthRepository) :
    AuthRefreshService {

  private val tasks = mutableMapOf<AuthTokens, Deferred<ApiResult<InternalAuthSession>>>()
  private val mutex = Mutex()

  override suspend fun refreshSession(
      authToken: String,
      refreshToken: String,
      currentLoginMethod: LoginMethod
  ): ApiResult<InternalAuthSession> = coroutineScope { // Added coroutineScope block
    val tokens = AuthTokens(authToken, refreshToken)

    // Mutex enforces one refresh task at a time
    mutex.withLock {
      tasks[tokens]?.await()?.let {
        // A refresh task with the same request tokens already created, so await it's response and
        // return
        return@coroutineScope it
      }

      val refreshTask = async {
        authRepository.refreshSession(
            authToken = authToken,
            refreshToken = refreshToken,
            currentLoginMethod = currentLoginMethod)
      }

      tasks[tokens] = refreshTask

      val result = refreshTask.await()

      result.map {
        val sameAuthTokensReturned =
            it.accessToken == tokens.accessToken && it.refreshToken == tokens.refreshToken
        // if response returns sessionUpdateAction == ignore, or if response returns same tokens as
        // request,
        // we don't want to add the response to the local cache
        if (it.sessionUpdateAction == SessionUpdateAction.Ignore || sameAuthTokensReturned) {
          tasks.remove(tokens)
        }

        it
      }

      result
    }
  }
}

private data class AuthTokens(val accessToken: String, val refreshToken: String)
