package io.privy.auth

import io.ktor.resources.Resource
import io.privy.auth.LoginType.CustomAccessToken
import io.privy.auth.internal.InternalAuthSession
import io.privy.auth.internal.InternalPrivyUser
import io.privy.auth.internal.InternalSiweMessage
import io.privy.auth.internal.LoginMethod
import io.privy.auth.otp.OtpRequestType
import io.privy.auth.session.AuthSessionResponseDeserializer
import io.privy.auth.session.internal.MapAuthSessionResponseToInternalAuthSession
import io.privy.auth.session.internal.toInternalPrivyUser
import io.privy.auth.siwe.toInternalSiweMessage
import io.privy.auth.siwe.GenerateSiweMessageResponse
import io.privy.auth.siwe.SiweLoginParams
import io.privy.logging.PrivyLogger
import io.privy.network.ApiResult
import io.privy.network.AuthType
import io.privy.network.KtorWrapper
import io.privy.network.map
import kotlinx.serialization.Serializable
import me.tatarka.inject.annotations.Inject

@Inject
public class RealAuthRepository(
    private val ktorWrapper: KtorWrapper,
    private val privyLogger: PrivyLogger,
    private val authSessionResponseDeserializer: AuthSessionResponseDeserializer,
    private val mapAuthSessionResponseToInternalAuthSession:
        MapAuthSessionResponseToInternalAuthSession,
) : AuthRepository {
  override suspend fun authenticate(loginType: LoginType): ApiResult<InternalAuthSession> {
    val authenticateRequest = AuthenticateResource(loginTypeApiPath = loginType.apiPath())

    val authenticateResponse: ApiResult<String> =
        ktorWrapper.postResult(
            resource = authenticateRequest,
            body = loginType.requestBody(),
            authType = AuthType.None,
        )

    return convertAuthResponseToInternalSession(
        authResponse = authenticateResponse,
        loginMethod = loginType.toLoginMethod(),
    )
  }

  override suspend fun link(loginType: LoginType, authToken: String): ApiResult<InternalPrivyUser> {
    return ktorWrapper
        .postResult<LinkAccountResource, Map<String, String>, String>(
            resource = LinkAccountResource(loginTypeApiPath = loginType.apiPath()),
            body = loginType.requestBody(),
            authType = AuthType.Authorization(accessToken = authToken),
        )
        .map { jsonResponse ->
          authSessionResponseDeserializer.deserializeRawUser(jsonResponse = jsonResponse)
        }
        .map { it.toInternalPrivyUser() }
  }

  override suspend fun generateSiweMessage(walletAddress: String): ApiResult<InternalSiweMessage> {

    return ktorWrapper
        .postResult<SiweInitResource, Map<String, String>, GenerateSiweMessageResponse>(
            resource = SiweInitResource,
            body = mapOf("address" to walletAddress),
            authType = AuthType.None,
        )
        .map { it.toInternalSiweMessage() }
  }

  override suspend fun refreshSession(
      authToken: String,
      refreshToken: String,
      currentLoginMethod: LoginMethod,
  ): ApiResult<InternalAuthSession> {
    val refreshResponse: ApiResult<String> =
        ktorWrapper.postResult(
            resource = RefreshSessionResource,
            body = mapOf("refresh_token" to refreshToken),
            authType = AuthType.Authorization(accessToken = authToken),
        )

    return convertAuthResponseToInternalSession(
        authResponse = refreshResponse,
        loginMethod = currentLoginMethod,
    )
  }

  override suspend fun refreshUser(authToken: String): ApiResult<InternalPrivyUser> {
    return ktorWrapper
        .getResult<RefreshUserResource, String>(
            resource = RefreshUserResource,
            authType = AuthType.Authorization(accessToken = authToken),
        )
        .map { jsonResponse ->
          privyLogger.internal("users/me response:\n$jsonResponse")
          authSessionResponseDeserializer.deserializeUser(jsonResponse = jsonResponse)
        }
        .map { it.user.toInternalPrivyUser() }
  }

  private fun convertAuthResponseToInternalSession(
      authResponse: ApiResult<String>,
      loginMethod: LoginMethod
  ): ApiResult<InternalAuthSession> {
    val mappedResponse =
        authResponse.map { jsonResponse ->
          privyLogger.internal("Authenticate API raw JSON response:\n$jsonResponse")

          authSessionResponseDeserializer.deserialize(jsonResponse = jsonResponse)
        }

    privyLogger.internal("Authenticate API response: $mappedResponse")

    val internalAuthSession =
        mappedResponse.map {
          mapAuthSessionResponseToInternalAuthSession.map(
              authSessionResponse = it,
              loginMethod = loginMethod,
          )
        }

    return internalAuthSession
  }

  override suspend fun sendOtp(otpRequestType: OtpRequestType): ApiResult<Boolean> {
    val sendOtpRequest = SendOtpResource(otpTypeApiPath = otpRequestType.apiPath())

    val sendOtpResponse: ApiResult<OtpInitResponse> =
        ktorWrapper.postResult(
            resource = sendOtpRequest,
            body = otpRequestType.requestBody(),
            authType = AuthType.None,
        )

    return sendOtpResponse.map { it.success }
  }
}

private fun LoginType.toLoginMethod(): LoginMethod {
  return when (this) {
    is LoginType.CustomAccessToken -> LoginMethod.CustomAccessToken
    is LoginType.Sms -> LoginMethod.Sms
    is LoginType.Email -> LoginMethod.Email
    is LoginType.Siwe -> LoginMethod.Siwe
  }
}

@Resource("{otpTypeApiPath}/init")
private class SendOtpResource(
    val otpTypeApiPath: String,
)

@Resource("{loginTypeApiPath}/authenticate")
private class AuthenticateResource(
    val loginTypeApiPath: String,
)

@Resource("{loginTypeApiPath}/link")
private class LinkAccountResource(
    val loginTypeApiPath: String,
)

@Resource("siwe/init") private data object SiweInitResource

@Resource("sessions") private data object RefreshSessionResource

@Resource("users/me") private data object RefreshUserResource

private fun LoginType.apiPath(): String {
  return when (this) {
    is CustomAccessToken -> "custom_jwt_account"
    is LoginType.Sms -> "passwordless_sms"
    is LoginType.Email -> "passwordless"
    is LoginType.Siwe -> "siwe"
  }
}

private fun LoginType.requestBody(): Map<String, String> {
  return when (this) {
    is CustomAccessToken -> mapOf("token" to this.token)
    is LoginType.Sms -> mapOf("phoneNumber" to this.phoneNumber, "code" to this.code)
    is LoginType.Email -> mapOf("email" to this.emailAddress, "code" to this.code)
    is LoginType.Siwe -> params.toRequestBody()
  }
}

private fun OtpRequestType.apiPath(): String {
  return when (this) {
    is OtpRequestType.Email -> "passwordless"
    is OtpRequestType.Phone -> "passwordless_sms"
  }
}

private fun OtpRequestType.requestBody(): Map<String, String> {
  return when (this) {
    is OtpRequestType.Email -> mapOf("email" to this.email)
    is OtpRequestType.Phone -> mapOf("phoneNumber" to this.phone)
  }
}

private fun SiweLoginParams.toRequestBody(): Map<String, String> {
  val body = mutableMapOf("message" to message, "signature" to signature, "chainId" to chainId)
  walletClientType?.let { body["walletClientType"] = it.toString() }
  connectorType?.let { body["connectorType"] = it }
  return body
}

@Serializable private data class OtpInitResponse(val success: Boolean)
