package io.privy.auth

import io.privy.auth.customAuth.TokenProvider
import io.privy.auth.internal.InternalAuthSession
import io.privy.auth.internal.InternalAuthState
import io.privy.auth.internal.InternalPrivyUser
import io.privy.auth.internal.InternalSiweMessage
import io.privy.auth.internal.LoginMethod
import io.privy.auth.jwt.DecodeJwt
import io.privy.auth.oAuth.OAuthInitRequest
import io.privy.auth.oAuth.OAuthInitResponse
import io.privy.auth.otp.OtpRequestType
import io.privy.auth.persistence.InternalAuthSessionTokens
import io.privy.auth.session.internal.SessionUpdateAction
import io.privy.auth.session.internal.AuthSessionResponse
import io.privy.auth.session.internal.AuthSessionResponseTokens
import io.privy.logging.PrivyLogger
import io.privy.network.toResult
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.IO
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.launch
import me.tatarka.inject.annotations.Inject

@Inject
public class RealAuthManager(
    private val authRepository: AuthRepository,
    private val authStateRepository: InternalAuthStateRepository,
    private val privyLogger: PrivyLogger,
    private val decodeJwt: DecodeJwt,
    private val authRefreshService: AuthRefreshService,
    privyUserCreator: (AuthManager) -> PrivyUser
) : AuthManager {
  private val scope = CoroutineScope(SupervisorJob() + Dispatchers.IO)

  // Create PrivyUser for internal use, but ONLY expose it publicly if there is an authenticated
  // session
  override val privyUser: PrivyUser = privyUserCreator(this)

  // Custom auth
  private var tokenProvider: TokenProvider? = null

  override fun setTokenProvider(tokenProvider: TokenProvider) {
    this.tokenProvider = tokenProvider
  }

  init {
    this.restoreSession()
  }

  /**
   * Generates an OAuth URL for initiating the authentication flow.
   * Communicates with Privy backend to create a URL with necessary OAuth parameters.
   */
  override suspend fun generateOAuthUrl(
    oAuthInitRequest: OAuthInitRequest
  ): Result<OAuthInitResponse> {
    return authRepository
      .generateOAuthUrl(oAuthInitRequest)
      .toResult()
  }

  /**
   * When the SDK is initialized, auth state is set to "NotReady" Privy is not ready to use until
   * auth state gets updated to something other than "NotReady". Thus, restoring session MUST update
   * auth state to authenticated or unauthenticated
   */
  private fun restoreSession() {
    scope.launch {
      privyLogger.internal("Attempting to restore prior session.")

      // Check if prior auth session exists
      val priorSessionTokens = authStateRepository.loadPersistedSessionTokens()

      privyLogger.error("PRIOR SESSION TOKENS: $priorSessionTokens")

      if (priorSessionTokens != null) {
        privyLogger.internal("Prior session exists!")

        // as part of restoring, if tokens are available, we should refresh session to fetch user
        priorSessionTokens.refreshSession()
      } else {
        privyLogger.internal("No prior session, user is unauthenticated!")
        // no prior session - trigger logout flow which will clear all data and set user to
        // unauthenticated
        logout()
      }
    }
  }

  // Links an additional account to the authenticated user
  override suspend fun linkAccount(loginType: LoginType): Result<Unit> {
    // Ensure session is valid before linking
    return refreshSessionIfNeeded().flatMap { authSession ->
      // Call repository to link account with current access token
      authRepository.link(loginType, authToken = authSession.accessToken).toResult().map {
          updatedUser ->
        // Update session with new user data after successful linking
        updateAuthSession(authSession.copy(user = updatedUser))
      }
    }
  }

  override suspend fun generateSiweMessage(walletAddress: String): Result<InternalSiweMessage> {
    return authRepository.generateSiweMessage(walletAddress).toResult()
  }

  /**
   * Publicly exposed login method, which converts internal auth session to user for convenience.
   */
  override suspend fun login(loginType: LoginType): Result<PrivyUser> {
    return internalLogin(loginType).map {
      // on successful login, return privy user
      privyUser
    }
  }

  override suspend fun sendOtp(otpRequestType: OtpRequestType): Result<Unit> {
    val result = authRepository.sendOtp(otpRequestType).toResult()

    return result.fold(
        onSuccess = { success ->
          if (success) {
            Result.success(Unit)
          } else {
            Result.failure(AuthenticationException("Error sending OTP."))
          }
        },
        onFailure = {
          // pass back the failure
          Result.failure(it)
        },
    )
  }

  private suspend fun internalLogin(loginType: LoginType): Result<InternalAuthSession> {
    val authResult = authRepository.authenticate(loginType = loginType).toResult().onFailure {
          privyLogger.internal("Login failed, error: $it")
        }

    return handleAuthResponse(authResult = authResult)
  }

  // Helper method that takes all required actions when we get a new auth response
  private suspend fun handleAuthResponse(
      authResult: Result<AuthSessionResponse>
  ): Result<InternalAuthSession> {
    return authResult.fold(
        onSuccess = { authSessionResponse ->
            handleSuccessfulAuthResponse(authSessionResponse)
        },
        onFailure = {
          // if auth response failed, clear user session if it existed
          this.logout()

          // Propagate error back up
          Result.failure(it)
        }
    )
  }

  private suspend fun handleSuccessfulAuthResponse(
    authSessionResponse: AuthSessionResponse
  ): Result<InternalAuthSession> {
    return when (val sessionUpdateAction = authSessionResponse.sessionUpdateAction) {
      // If we receive a "clear" update action, log the user out
      is SessionUpdateAction.Clear -> handleClearAuthSession()
      is SessionUpdateAction.Ignore -> handleIgnoreAuthSession(
        user = authSessionResponse.user,
        tokens = sessionUpdateAction.authSessionResponseTokens,
        loginMethod = authSessionResponse.loginMethod,
      )
      is SessionUpdateAction.Set -> handleSetAuthSession(
        user = authSessionResponse.user,
        tokens = sessionUpdateAction.authSessionResponseTokens,
        loginMethod = authSessionResponse.loginMethod,
      )
    }
  }

  private suspend fun handleSetAuthSession(
    user: InternalPrivyUser,
    tokens: AuthSessionResponseTokens,
    loginMethod: LoginMethod,
  ): Result<InternalAuthSession> {
    val newInternalAuthSession = createInternalAuthSession(
      user = user,
      tokens = tokens,
      loginMethod = loginMethod,
    )

    updateAuthSession(newInternalAuthSession)

    return Result.success(newInternalAuthSession)
  }

  private suspend fun handleClearAuthSession(): Result<InternalAuthSession> {
    // If backend specified "clear" for "sessionUpdateAction", log the user out
    logout()

    // Propagate up as an error since we consider this unsuccessful authenticate attempt
    return Result.failure(
      AuthenticationException("Authenticate was successful, but Privy backend requested this user be logged out.")
    )
  }

  private suspend fun handleIgnoreAuthSession(
    user: InternalPrivyUser,
    tokens: AuthSessionResponseTokens?,
    loginMethod: LoginMethod,
  ): Result<InternalAuthSession> {
    // In the case of ignore, we may still be in a state where session tokens have not yet been set.
    return if (tokens != null) {
      // If the token is present and non-empty, we attempt to set it
      handleSetAuthSession(
        user = user,
        tokens = tokens,
        loginMethod = loginMethod,
      )
    } else {
      // If tokens are not present, update the user of the current session, if there is one
      val currAuthState = authStateRepository.internalAuthState.value

      return if (currAuthState is InternalAuthState.Authenticated) {
        val updatedSession = currAuthState.session.copyAndUpdateUser(user)
        Result.success(updatedSession)
      } else {
        Result.failure(
          AuthenticationException("User must be authenticated before calling refresh user."))
      }
    }
  }

  private fun createInternalAuthSession(
    user: InternalPrivyUser,
    tokens: AuthSessionResponseTokens,
    loginMethod: LoginMethod,
  ): InternalAuthSession {
    return InternalAuthSession(
      user = user,
      accessToken = tokens.token,
      refreshToken = tokens.refreshToken,
      identityToken = tokens.identityToken,
      loginMethod = loginMethod,
    )
  }

  public override fun <T> authenticatedOrDefault(
      onAuthenticated: (InternalAuthSession) -> T,
      default: () -> T
  ): T {
    // Current session
    val currAuthState = authStateRepository.internalAuthState.value

    return if (currAuthState is InternalAuthState.Authenticated) {
      onAuthenticated(currAuthState.session)
    } else {
      default()
    }
  }

  // Helper function to ensure user is authenticated before making wallet call
  // If authenticated: returns result.success with associated session
  // If unauthenticated: returns result.failure
  public override suspend fun <T> ensureAuthenticated(
      onAuthenticated: suspend (InternalAuthSession) -> Result<T>
  ): Result<T> {
    return refreshSessionIfNeeded()
        .fold(onSuccess = { onAuthenticated(it) }, onFailure = { Result.failure(it) })
  }

  override suspend fun refreshUser(): Result<InternalAuthSession> {
    // Current session
    val currAuthState = authStateRepository.internalAuthState.value

    return if (currAuthState is InternalAuthState.Authenticated) {
      currAuthState.session.refreshUser()
    } else {
      Result.failure(
          AuthenticationException("User must be authenticated before calling refresh user."))
    }
  }

  override suspend fun refreshSession(): Result<InternalAuthSession> {
    // Current session
    val currAuthState = authStateRepository.internalAuthState.value

    return if (currAuthState is InternalAuthState.Authenticated) {
      currAuthState.session.refreshSession()
    } else {
      Result.failure(AuthenticationException("User must be authenticated before calling refresh."))
    }
  }

  override suspend fun logout() {
    authStateRepository.updateInternalAuthState(
        internalAuthState = InternalAuthState.Unauthenticated,
    )
  }

  override suspend fun refreshSessionIfNeeded(): Result<InternalAuthSession> {
    // Current session
    val currAuthState = authStateRepository.internalAuthState.value

    return if (currAuthState is InternalAuthState.Authenticated) {
      currAuthState.session.refreshSessionIfNeeded()
    } else {
      Result.failure(
          AuthenticationException("User must be authenticated before calling refresh session."))
    }
  }

  private suspend fun updateAuthSession(authSession: InternalAuthSession) {
    // user is authenticated - update auth state
    // Note: external auth state should automatically update after setting internal state.
    authStateRepository.updateInternalAuthState(
        internalAuthState = InternalAuthState.Authenticated(session = authSession),
    )
  }

  private suspend fun InternalAuthSession.refreshSessionIfNeeded(): Result<InternalAuthSession> {
    val jwt = decodeJwt(this.accessToken)

    return if (jwt.isExpired()) {
      // If session is expired, try refreshing
      this.refreshSession()
    } else {
      // If session is already valid, return itself
      Result.success(this)
    }
  }

  // Helper function to refresh user in the scope of an existing session
  private suspend fun InternalAuthSession.refreshUser(): Result<InternalAuthSession> {
    val jwt = decodeJwt(this.accessToken)

    return if (jwt.isExpired()) {
      // If session is expired, refresh it, which will also refresh user
      this.refreshSession()
    } else {
      // If session is already valid, just refresh the user
      return authRepository
          .refreshUser(this.accessToken)
          .toResult()
          .fold(
              onSuccess = { refreshedUser ->
                val updatedSession = copyAndUpdateUser(refreshedUser)
                Result.success(updatedSession)
              },
              onFailure = { Result.failure(it) })
    }
  }

  /**
   * Updates the user object in the current auth session.
   * @return The updated auth session
   */
  private suspend fun InternalAuthSession.copyAndUpdateUser(updatedUser: InternalPrivyUser): InternalAuthSession {
    val updatedSession = this.copy(user = updatedUser)
    updateAuthSession(updatedSession)
    return updatedSession
  }

  // Helper function to refresh user in the scope of an existing session
  private suspend fun InternalAuthSession.refreshSession(): Result<InternalAuthSession> {
    return when (this.loginMethod) {
      LoginMethod.CustomAccessToken -> reAuthenticateWithCustomAuth()
      else -> {
        return refreshTokensWithPrivy(
            currentAccessToken = this.accessToken,
            currentRefreshToken = this.refreshToken,
            currentLoginMethod = this.loginMethod,
          )
      }
    }
  }

  private suspend fun InternalAuthSessionTokens.refreshSession(): Result<InternalAuthSession> {
    return when (this.loginMethod) {
      LoginMethod.CustomAccessToken -> reAuthenticateWithCustomAuth()
      else -> {
        return refreshTokensWithPrivy(
            currentAccessToken = this.accessToken,
            currentRefreshToken = this.refreshToken,
            currentLoginMethod = this.loginMethod)
      }
    }
  }

  private suspend fun refreshTokensWithPrivy(
      currentAccessToken: String,
      currentRefreshToken: String,
      currentLoginMethod: LoginMethod
  ): Result<InternalAuthSession> {
    // Using auth refresh service guarantees thread safety
    val refreshResult =
        authRefreshService
            .refreshSession(
                authToken = currentAccessToken,
                refreshToken = currentRefreshToken,
                currentLoginMethod = currentLoginMethod,
              )
            .toResult()

    return handleAuthResponse(authResult = refreshResult)
  }

  private suspend fun reAuthenticateWithCustomAuth(): Result<InternalAuthSession> {
    val customAuthAccessToken = tokenProvider?.invoke()

    if (customAuthAccessToken == null) {
      // since reauthenticating user failed, log them out
      privyLogger.debug("Custom access token from tokenProvider null. Logging user out.")
      this.logout()
      return Result.failure(
          AuthenticationException("Custom access token from tokenProvider null. Logging user out."),
      )
    }

    return this.internalLogin(
        loginType = LoginType.CustomAccessToken(token = customAuthAccessToken),
    )
  }
}
