package io.privy.auth.email

import io.privy.auth.AuthManager
import io.privy.auth.AuthenticationException
import io.privy.auth.LoginType
import io.privy.auth.PrivyUser
import io.privy.auth.otp.OtpRequestType
import me.tatarka.inject.annotations.Inject

@Inject
public class RealLoginWithEmail(private val authManager: AuthManager) : LoginWithEmail {
  private var emailOtpSentTo: String? = null

  override suspend fun sendCode(email: String): Result<Unit> {
    return authManager.sendOtp(otpRequestType = OtpRequestType.Email(email)).onSuccess {
      // Cache email OTP was sent to
      this.emailOtpSentTo = email
    }
  }

  override suspend fun loginWithCode(code: String, email: String?): Result<PrivyUser> {
    // if email is specified in method parameters, use it, else fall back to cached
    val sentToEmail =
        email
            ?: emailOtpSentTo
                ?: return Result.failure(AuthenticationException("Email address not specified."))

    return authManager.login(loginType = LoginType.Email(emailAddress = sentToEmail, code = code))
  }

  override suspend fun linkWithCode(code: String, email: String?): Result<Unit> {
    val sentToEmail =
        email
            ?: emailOtpSentTo
                ?: return Result.failure(AuthenticationException("Email address not specified."))

    return authManager.linkAccount(
        loginType = LoginType.Email(emailAddress = sentToEmail, code = code))
  }
}
