package io.privy.auth.siwe

import io.privy.auth.AuthManager
import io.privy.auth.LoginType
import io.privy.auth.PrivyUser
import io.privy.auth.internal.InternalSiweMessage
import kotlinx.datetime.Clock
import me.tatarka.inject.annotations.Inject

@Inject
public class RealLoginWithSiwe(private val authManager: AuthManager) : LoginWithSiwe {

  private object Constants {
    const val SIWE_MESSAGE_STATEMENT =
        "By signing, you are proving you own this wallet and logging in. This does not initiate a transaction or cost any fees."
  }

  override suspend fun generateSiweMessage(
      params: SiweMessageParams,
      metadata: WalletLoginMetadata?
  ): Result<String> {
    return authManager.generateSiweMessage(params.walletAddress).map { response ->
      createSiweMessage(params, response)
    }
  }

  override suspend fun loginWithSiwe(
      message: String,
      signature: String,
      params: SiweMessageParams,
      metadata: WalletLoginMetadata?,
  ): Result<PrivyUser> {
    val result =
        executeSiweAuth(message, signature, params, metadata) { loginType ->
          authManager.login(loginType = loginType)
        }

    return result;
  }

    override suspend fun linkWithSiwe(
        message: String,
        signature: String,
        params: SiweMessageParams,
        metadata: WalletLoginMetadata?,
    ): Result<Unit> {
        // Ensure user is authenticated, then proceed with linking if successful
        return authManager.ensureAuthenticated { _ ->
            executeSiweAuth(message, signature, params, metadata) { loginType ->
                authManager.linkAccount(loginType = loginType)
            }
        }
    }

  private fun createSiweMessage(params: SiweMessageParams, response: InternalSiweMessage): String {
    val issuedAt = Clock.System.now().toString()
    val message =
        """
        ${params.appDomain} wants you to sign in with your Ethereum account:
        ${params.walletAddress}

        ${Constants.SIWE_MESSAGE_STATEMENT}

        URI: ${params.appUri}
        Version: 1
        Chain ID: ${params.chainId}
        Nonce: ${response.nonce}
        Issued At: $issuedAt
        Resources:
        - https://privy.io
        """
            .trimIndent()

    return message
  }

  private suspend fun <T> executeSiweAuth(
      message: String,
      signature: String,
      params: SiweMessageParams,
      metadata: WalletLoginMetadata?,
      authCall: suspend (LoginType) -> T
  ): T {

    val siweLoginType =
        LoginType.Siwe(
            params =
                SiweLoginParams(
                    message = message,
                    signature = signature,
                    chainId = params.chainId,
                    walletClientType = metadata?.walletClientType,
                    connectorType = metadata?.connectorType,
                ))

    return authCall(siweLoginType)
  }
}
