package io.privy.auth.oAuth

import io.privy.auth.AuthManager
import io.privy.auth.LoginType
import io.privy.auth.PrivyUser
import io.privy.logging.PrivyLogger
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.IO
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.launch
import io.ktor.http.Url
import me.tatarka.inject.annotations.Inject
import kotlin.Result

/**
 * Implementation of the LoginWithOAuth interface that delegates to an OAuthHandler.
 * This class serves as a bridge between the public API and the internal OAuth handling mechanism,
 * providing a clean separation of concerns in the authentication system.
 */
@Inject
public class RealLoginWithOAuth(
    private val oAuthHandler: OAuthHandler,
    private val pkceHelper: PKCEHelper,
    private val authManager: AuthManager,
    private val logger: PrivyLogger,
) : LoginWithOAuth {
    // Coroutine scope for processing OAuth responses asynchronously
    private val processingScope = CoroutineScope(SupervisorJob() + Dispatchers.IO)

    // Map to store pending OAuth requests by state code
    private val pendingStates = mutableMapOf<String, PendingOAuthInfoInternal>()

    /**
     * Initiates the OAuth login flow by delegating to the OAuthHandler.
     * This method simply forwards the request to the underlying OAuthHandler implementation,
     * which handles the actual OAuth authentication process.
     *
     * @param oAuthProvider The OAuth provider to authenticate with
     * @param appUrlScheme The URL scheme for handling redirects
     * @param onComplete Callback to be invoked with the result of the login attempt
     */
    override suspend fun login(
        oAuthProvider: OAuthProvider,
        appUrlScheme: String,
        onComplete: (Result<PrivyUser>) -> Unit
    ) {
        // Validate that app URL scheme is provided, as it's required for the redirect
        if (appUrlScheme.isEmpty()) {
            onComplete(Result.failure(IllegalArgumentException("App URL scheme must be provided")))
            return
        }

        // Clear any pending states from abandoned OAuth flows
        pendingStates.clear()

        // Generate PKCE (Proof Key for Code Exchange) parameters for secure authorization
        logger.debug("Generating PKCE parameters for OAuth provider: $oAuthProvider")
        val codeVerifier = pkceHelper.generateCodeVerifier()
        val codeChallenge = pkceHelper.deriveCodeChallenge(codeVerifier)
        val stateCode = pkceHelper.generateStateCode()

        // Create and store pending OAuth information for later verification
        val pendingInfo = PendingOAuthInfoInternal(
            state = stateCode,
            codeVerifier = codeVerifier,
            provider = oAuthProvider,
            onCompleteCallback = onComplete
        )

        storePendingInfo(stateCode, pendingInfo)

        // Launch the OAuth flow in the browser and handle any launch failures
        val launchResult = prepareAndLaunchInternal(
            oAuthProvider,
            appUrlScheme,
            codeChallenge,
            stateCode
        )

        if (launchResult.isFailure) {
          logger.error("Failed to launch OAuth browser for provider: $oAuthProvider")
            removePendingInfo(stateCode)
            onComplete(Result.failure(launchResult.exceptionOrNull() ?: Exception("Browser launch failed")))
        }
    }

    /**
     * Handles the redirect URL by delegating to the OAuthHandler.
     * This method forwards the redirect URL to the underlying OAuthHandler implementation,
     * which processes the OAuth response and completes the authentication flow.
     *
     * @param intentDataString The redirect URL to process
     * @return Boolean indicating whether the redirect was successfully handled
     */
    override fun handleIntentData(intentDataString: String): Boolean {
        val oAuthRedirectQueryParams = extractOAuthRedirectParamsOrNull(intentDataString) ?: kotlin.run{
            logger.error("Privy can't handle this oAuth redirect URL.")
            return false
        }

        // Retrieve and remove the pending OAuth information using the state parameter
        logger.debug("Processing OAuth redirect with state: ${oAuthRedirectQueryParams.state}")
        val pendingInfo = removePendingInfo(oAuthRedirectQueryParams.state) ?: run {
            logger.error("No pending OAuth request found for state: ${oAuthRedirectQueryParams.state}")
            return false
        }

        // Validate that the received state matches the expected state to prevent CSRF attacks
        if (oAuthRedirectQueryParams.state != pendingInfo.state) {
            logger.error("OAuth state mismatch. Received: ${oAuthRedirectQueryParams.state}, Expected: ${pendingInfo.state}")
            pendingInfo.onCompleteCallback(Result.failure(Exception("OAuth State Mismatch")))
            return false
        }
        
        logger.debug("OAuth state validation successful for provider: ${pendingInfo.provider}")

        // Asynchronously complete the OAuth flow by exchanging the authorization code for tokens
        logger.debug("Starting async OAuth code exchange for provider: ${pendingInfo.provider}")
        processingScope.launch {
            val result = try {
                authManager.login(
                    LoginType.OAuth(
                        provider = pendingInfo.provider,
                        authorizationCode = oAuthRedirectQueryParams.code,
                        codeVerifier = pendingInfo.codeVerifier,
                        stateCode = oAuthRedirectQueryParams.state
                    )
                )
            } catch (e: Exception) {
                logger.error("OAuth login failed", e)
                Result.failure(e)
            }
            // Invoke the callback with the login result
            if (result.isSuccess) {
                logger.debug("OAuth authentication succeeded for provider: ${pendingInfo.provider}")
            } else {
                logger.error("OAuth authentication failed for provider: ${pendingInfo.provider}")
            }
            pendingInfo.onCompleteCallback(result)
        }
        
        return true
    }

    override fun canHandleIntentData(intentDataString: String): Boolean {
        val oAuthRedirectQueryParams = extractOAuthRedirectParamsOrNull(intentDataString)
        val canHandle = oAuthRedirectQueryParams != null
        if (!canHandle) {
            logger.debug("URL is not a valid OAuth redirect: $intentDataString")
        }
        return canHandle
    }

    /**
     * Prepares and launches the OAuth authentication flow.
     * Generates the OAuth URL and opens it in a browser tab.
     *
     * @param oAuthProvider The OAuth provider to authenticate with
     * @param appUrlScheme The URL scheme for handling the redirect
     * @param codeChallenge The PKCE code challenge
     * @param stateCode The state code for verifying the response
     * @return Result indicating success or failure of launching the authentication flow
     */
    private suspend fun prepareAndLaunchInternal(
        oAuthProvider: OAuthProvider,
        appUrlScheme: String,
        codeChallenge: String,
        stateCode: String
    ): Result<Unit> {
        return try {
            authManager.generateOAuthUrl(
                oAuthInitRequest = OAuthInitRequest(
                    provider = oAuthProvider,
                    redirectUri = appUrlScheme.addSchemeSeparatorIfNeeded(),
                    codeChallenge = codeChallenge,
                    stateCode = stateCode
                )
            ).fold(
                onSuccess = { initData ->
                    oAuthHandler.launchCustomTab(initData.url)
                },
                onFailure = { error ->
                    Result.failure(error)
                }
            )
        } catch (e: Exception) {
            Result.failure(e)
        }
    }

    /**
     * Stores the pending OAuth information for later retrieval.
     * Thread-safe operation using synchronization.
     *
     * @param key The state code to use as the key
     * @param info The pending OAuth information to store
     */
    private fun storePendingInfo(key: String, info: PendingOAuthInfoInternal) {
        pendingStates[key] = info
    }

    /**
     * Removes and returns the pending OAuth information for the given key.
     * Thread-safe operation using synchronization.
     *
     * @param key The state code used as the key
     * @return The pending OAuth information, or null if not found
     */
    private fun removePendingInfo(key: String): PendingOAuthInfoInternal? {
        return pendingStates.remove(key)
    }

    /**
     * Adds the scheme separator (://) to the URL scheme if needed.
     * Ensures the URL scheme is properly formatted.
     *
     * @return The properly formatted URL scheme
     */
    private fun String.addSchemeSeparatorIfNeeded(): String {
        return if (!this.contains("://")) {
            if (this.matches("^[a-zA-Z][a-zA-Z0-9+.-]*$".toRegex())) "$this://" else this
        } else this
    }

    /**
     * Extracts OAuth redirect parameters from a URL or returns null if the URL is not a valid OAuth redirect.
     */
    private fun extractOAuthRedirectParamsOrNull(url: String): OAuthRedirectQueryParams? {
        if (url.isNullOrEmpty()) return null
        
        return try {
            val parsedUrl = Url(url)
            val params = parsedUrl.parameters
            
            val code = params["privy_oauth_code"]
            val state = params["privy_oauth_state"]

             if (code == null || state == null) return null
            
             OAuthRedirectQueryParams(code, state)
        } catch (_: Exception) {
             null
        }
    }
}