package io.privy.auth.internal

import io.privy.auth.LoginType
import io.privy.auth.PrivyUser
import io.privy.auth.customAuth.TokenProvider

// Serves as the main logic hub for the authentication module
public interface InternalAuthManager {
  public val privyUser: PrivyUser

  // For custom auth refresh flow
  public fun setTokenProvider(tokenProvider: TokenProvider)

  /**
   * Restores the prior user session, if it exists.
   *
   * @return Result.success, if session is restored, Result.failure otherwise, including
   * the case where there was no prior session to restore.
   */
  public suspend fun restorePriorSessionIfNeeded()

  public suspend fun login(loginType: LoginType): Result<PrivyUser>

  // Helper function to return a value if user is authenticated,
  // or else return the provided default value
  public fun <T> authenticatedOrDefault(
    onAuthenticated: (InternalAuthSession) -> T,
    default: () -> T
  ): T

  // Helper function to ensure user is authenticated before making wallet call
  // If authenticated: returns result.success with associated session
  // If unauthenticated: returns result.failure
  public suspend fun <T> ensureAuthenticated(
      onAuthenticated: suspend (InternalAuthSession) -> Result<T>
  ): Result<T>

  /**
   * Force refreshes existing session.
   *
   * @return Result.success if session was successfully refreshed or Result.failure if there is no current
   * session to refresh or if refresh fails
   */
  public suspend fun refreshSession(): Result<InternalAuthSession>

  public suspend fun linkAccount(loginType: LoginType): Result<Unit>

  public suspend fun updateAccount(loginType: LoginType): Result<Unit>

  /**
   * Checks if the current session is still valid. If the session is expired, it attempts to refresh
   * it.
   *
   * @return Result.success with a valid, refreshed session if needed, or the existing session if
   *   it's still valid. Result.failure if there is no current session to refresh or if the refresh
   *   fails.
   */
  public suspend fun refreshSessionIfNeeded(): Result<InternalAuthSession>

  /**
   * Refreshes the current session's user. If this session is expired, the session will be
   * refreshed.
   *
   * @return Result.success with a valid session, containing the updated user Result.failure if
   *   there is no current session to refresh or if refresh fails
   */
  public suspend fun refreshUser(): Result<InternalAuthSession>

  public suspend fun logout()

  public suspend fun awaitInitializationComplete()

  public suspend fun hasPersistedAuthCredentials(): Boolean
}
