package io.privy.auth.internal

import io.privy.auth.persistence.AuthSessionDataStore
import io.privy.auth.persistence.InternalAuthSessionTokens
import io.privy.di.KmpAppScope
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.firstOrNull
import kotlinx.coroutines.flow.update
import me.tatarka.inject.annotations.Inject

public interface InternalInternalAuthStateRepository : InternalAuthStateRepository {
  public suspend fun updateInternalAuthState(internalAuthState: InternalAuthState)

  public suspend fun loadPersistedSessionTokens(): InternalAuthSessionTokens?
}

@Inject
@KmpAppScope
public class RealInternalAuthStateRepository(
  private val authSessionDataStore: AuthSessionDataStore,
) : InternalInternalAuthStateRepository {
  // Define auth state state flow with NotReady as initial value
  private val _internalAuthState: MutableStateFlow<InternalAuthState> =
      MutableStateFlow(InternalAuthState.NotReady)

  override val internalAuthState: StateFlow<InternalAuthState> =
      this._internalAuthState.asStateFlow()

  override val currentAuthState: InternalAuthState
    get() = internalAuthState.value

  override suspend fun loadPersistedSessionTokens(): InternalAuthSessionTokens? {
    return authSessionDataStore.internalAuthSessionTokens.firstOrNull()
  }

  override suspend fun updateInternalAuthState(internalAuthState: InternalAuthState) {
    when (internalAuthState) {
      InternalAuthState.NotReady -> {}
      InternalAuthState.AuthenticatedUnverified -> {}
      is InternalAuthState.Authenticated -> {
        authSessionDataStore.saveAuthSession(internalAuthState.session)
      }
      InternalAuthState.Unauthenticated -> {
        authSessionDataStore.clearAuthSession()
      }
    }

    _internalAuthState.update { internalAuthState }
  }
}
