package io.privy.auth.internal

import io.privy.auth.AuthRefreshService
import io.privy.auth.AuthRepository
import io.privy.auth.AuthenticationException
import io.privy.auth.LoginType
import io.privy.auth.PrivyReadyStateManager
import io.privy.auth.PrivyUser
import io.privy.auth.UpdateAccountType
import io.privy.auth.customAuth.TokenProvider
import io.privy.auth.flatMap
import io.privy.auth.jwt.DecodeJwt
import io.privy.auth.persistence.InternalAuthSessionTokens
import io.privy.auth.session.internal.SessionUpdateAction
import io.privy.auth.session.internal.AuthSessionResponse
import io.privy.auth.session.internal.AuthSessionResponseTokens
import io.privy.logging.PrivyLogger
import io.privy.network.NetworkStateManager
import io.privy.network.failIfNetworkConfirmedDisconnectedElse
import io.privy.network.isConfirmedDisconnected
import io.privy.network.toResult
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.IO
import kotlinx.coroutines.Job
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.launch
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import me.tatarka.inject.annotations.Inject

@Inject
public class RealInternalAuthManager(
  private val authRepository: AuthRepository,
  private val authStateRepository: InternalInternalAuthStateRepository,
  private val privyLogger: PrivyLogger,
  private val decodeJwt: DecodeJwt,
  private val authRefreshService: AuthRefreshService,
  private val networkStateManager: NetworkStateManager,
  privyUserCreator: (InternalAuthManager) -> PrivyUser
) : InternalAuthManager {
  private val scope = CoroutineScope(SupervisorJob() + Dispatchers.IO)

  // Create PrivyUser for internal use, but ONLY expose it publicly if there is an authenticated
  // session
  override val privyUser: PrivyUser = privyUserCreator(this)

  // Custom auth
  private var tokenProvider: TokenProvider? = null

  private val restoreSessionMutex = Mutex()
  private var networkUpdatesJob: Job? = null
  private var privyReadyStateManager = PrivyReadyStateManager()

  override fun setTokenProvider(tokenProvider: TokenProvider) {
    this.tokenProvider = tokenProvider
  }

  init {
    scope.launch {
      if (networkStateManager.current.isConfirmedDisconnected()) {
        privyLogger.debug("Privy initialized with no network connectivity.")
        handleNetworkOfflineAtInit()
      } else {
        // Network is either connected or unknown, so attempt to restore session
        restorePriorSessionIfNeeded()
      }

      // Once either of the flows above complete, we can consider Privy ready to use
      // Accessing auth state after this will guarantee the most up to date value
      privyLogger.internal("Setting privy ready state to true")
      privyReadyStateManager.setReady(true)
    }
  }

  override suspend fun awaitInitializationComplete() {
    privyReadyStateManager.awaitReady()
    privyLogger.internal("Privy is ready")
  }

  override suspend fun hasPersistedAuthCredentials(): Boolean {
    val priorSessionTokens = authStateRepository.loadPersistedSessionTokens()
    return priorSessionTokens != null
  }

  /**
   * Handle Privy SDK init when network is offline
   */
  private suspend fun handleNetworkOfflineAtInit() {
    // Check if there is a prior session
    val priorSessionTokens = authStateRepository.loadPersistedSessionTokens()

    if (priorSessionTokens == null) {
      // If there is no prior session, we can automatically set the user state to Unauthenticated
      privyLogger.debug("No prior session -- setting auth state to Unauthenticated")
      logout()
    } else {
      // If there is a prior session, we can't restore without network.
      // Print a log to tell devs how to handle it.
      privyLogger.debug(
        """
          A prior user session exists, but Privy can't verify the authenticated state of the user
          until network connectivity is restored. The auth state will remain "AuthenticatedUnverified" until
          connectivity is restored, at which point, Privy will automatically attempt to confirm
          the user's authenticated state.
          
          Alternatively, you may call Privy.onNetworkRestored once you determine network is restored.
        """.trimIndent()
      )

      authStateRepository.updateInternalAuthState(
        internalAuthState = InternalAuthState.AuthenticatedUnverified,
      )

      // subscribe to network updates so we can restore session when network returns
      subscribeToNetworkRestoredUpdates()
    }
  }

  /**
   * Attempts to restore the user's prior session if users auth state has not already been determined.
   * If no prior session exists, we clear out any user state and set the state to Unauthenticated.
   * Important: The caller of this function should have already checked network state. If this function
   * is called with no network, the session restoration will fail and set the auth state to Unauthenticated
   */
  override suspend fun restorePriorSessionIfNeeded() {
    // We only want to attempt restoring session once a time, so use mutex
    restoreSessionMutex.withLock {
      // Only restore a session if auth state isn't already determined
      if (authStateRepository.internalAuthState.value.hasBeenDetermined()) {
        privyLogger.debug("User auth state already determined. No need to restore.")
        return
      }

      // Check if prior auth session exists
      val priorSessionTokens = authStateRepository.loadPersistedSessionTokens()

      if (priorSessionTokens != null) {
        privyLogger.debug("Prior session exists! Attempting to restore")

        // as part of restoring, if tokens are available, we should refresh session to fetch user
        priorSessionTokens
          .refreshSession()
          .fold(
            onSuccess = {
              // refreshSession internally handles setting auth state
              privyLogger.debug("Prior session successfully restored.")
            },
            onFailure = {
              // In the case of session restoration, if refresh session fails due to network connectivity,
              // we want to explictly set auth state to Unauthenticated, thus we call logout
              privyLogger.debug("Failed to restore prior session, settings auth state to Unauthenticated.")
              logout()
            }
          )
      } else {
        privyLogger.internal("No prior session to restore, setting auth state to Unauthenticated.")

        // Log user out to clear state and set auth state to Unauthenticated
        logout()
      }
    }
  }

  // Links an additional account to the authenticated user
  override suspend fun linkAccount(loginType: LoginType): Result<Unit> {
    // Ensure session is valid before linking
    return refreshSessionIfNeeded().flatMap { authSession ->
      // Call repository to link account with current access token
      authRepository.link(loginType, authToken = authSession.accessToken).toResult().map {
          updatedUser ->
        // Update session with new user data after successful linking
        updateAuthSession(authSession.copy(user = updatedUser))
      }
    }
  }

  // Updates the specified account for the user
  override suspend fun updateAccount(loginType: LoginType): Result<Unit> {
    // Ensure session is valid before linking
    return refreshSessionIfNeeded().flatMap { authSession ->
      loginType.toUpdateAccountType(authSession).flatMap { updateAccountType ->
        authRepository.update(updateAccountType, authToken = authSession.accessToken).toResult().map { updatedUser ->
          // Update session with new user data after successful linking
          updateAuthSession(authSession.copy(user = updatedUser))
        }
      }
    }
  }

  private fun LoginType.toUpdateAccountType(authSession: InternalAuthSession): Result<UpdateAccountType> {
    return when (this) {
      is LoginType.Email -> {
        // ensure user has an exisiting email
        val currentUserEmail = authSession.user.emailAddress
          ?: return Result.failure(
            AuthenticationException("User must have an existing email linked to update it.")
          )

        // User has an existing email
        return Result.success(
            UpdateAccountType.Email(
                oldEmail = currentUserEmail,
                newEmail = this.emailAddress,
                code = this.code,
            ))
      }
      is LoginType.Sms -> {
        // ensure user has an existing phone number
        val currentUserPhoneNumber =
            authSession.user.phoneNumber
                ?: return Result.failure(
                    AuthenticationException(
                        "User must have an existing phone number linked to update it."))

        // User has an existing phone number
        Result.success(
            UpdateAccountType.Sms(
                oldPhoneNumber = currentUserPhoneNumber,
                newPhoneNumber = this.phoneNumber,
                code = this.code,
            ))
      }
      else -> {
        // Currently don't support update for the login type
        Result.failure(
          AuthenticationException("Email address not specified.")
        )
      }
    }
  }

  /**
   * Publicly exposed login method, which converts internal auth session to user for convenience.
   */
  override suspend fun login(loginType: LoginType): Result<PrivyUser> {
    return internalLogin(loginType).map {
      // on successful login, return privy user
      privyUser
    }
  }

  private fun subscribeToNetworkRestoredUpdates() {
    networkUpdatesJob = scope.launch {
      networkStateManager
        .observeNetworkRestored()
        .collectLatest {
          // If network is regained, and auth state has not yet been determined,
          // restore prior session if needed
          if (!authStateRepository.currentAuthState.hasBeenDetermined()) {
            privyLogger.debug("Automatically determining user's auth state because network has been restored and auth state had not yet been determined.")

            restorePriorSessionIfNeeded()

            // now that we've successfully attempting to restore session, cancel network updates listener
            networkUpdatesJob?.cancel()
            networkUpdatesJob = null
          }
        }
    }
  }

  private suspend fun internalLogin(loginType: LoginType): Result<InternalAuthSession> {
    val authResult = authRepository.authenticate(loginType = loginType).toResult().onFailure {
      privyLogger.internal("Login failed, error: $it")
    }

    return handleAuthResponse(authResult = authResult)
  }

  // Helper method that takes all required actions when we get a new auth response
  private suspend fun handleAuthResponse(
    authResult: Result<AuthSessionResponse>
  ): Result<InternalAuthSession> {
    return authResult.fold(
      onSuccess = { authSessionResponse ->
        handleSuccessfulAuthResponse(authSessionResponse)
      },
      onFailure = {
        // if auth response failed, clear user session if it existed
        this.logout()

        // Propagate error back up
        Result.failure(it)
      }
    )
  }

  private suspend fun handleSuccessfulAuthResponse(
    authSessionResponse: AuthSessionResponse
  ): Result<InternalAuthSession> {
    return when (val sessionUpdateAction = authSessionResponse.sessionUpdateAction) {
      // If we receive a "clear" update action, log the user out
      is SessionUpdateAction.Clear -> handleClearAuthSession()
      is SessionUpdateAction.Ignore -> handleIgnoreAuthSession(
        user = authSessionResponse.user,
        tokens = sessionUpdateAction.authSessionResponseTokens,
        loginMethod = authSessionResponse.loginMethod,
      )
      is SessionUpdateAction.Set -> handleSetAuthSession(
        user = authSessionResponse.user,
        tokens = sessionUpdateAction.authSessionResponseTokens,
        loginMethod = authSessionResponse.loginMethod,
      )
    }
  }

  private suspend fun handleSetAuthSession(
    user: InternalPrivyUser,
    tokens: AuthSessionResponseTokens,
    loginMethod: LoginMethod,
  ): Result<InternalAuthSession> {
    val newInternalAuthSession = createInternalAuthSession(
      user = user,
      tokens = tokens,
      loginMethod = loginMethod,
    )

    updateAuthSession(newInternalAuthSession)

    return Result.success(newInternalAuthSession)
  }

  private suspend fun handleClearAuthSession(): Result<InternalAuthSession> {
    // If backend specified "clear" for "sessionUpdateAction", log the user out
    logout()

    // Propagate up as an error since we consider this unsuccessful authenticate attempt
    return Result.failure(
      AuthenticationException("Authenticate was successful, but Privy backend requested this user be logged out.")
    )
  }

  private suspend fun handleIgnoreAuthSession(
    user: InternalPrivyUser,
    tokens: AuthSessionResponseTokens?,
    loginMethod: LoginMethod,
  ): Result<InternalAuthSession> {
    // In the case of ignore, we may still be in a state where session tokens have not yet been set.
    return if (tokens != null) {
      // If the token is present and non-empty, we attempt to set it
      handleSetAuthSession(
        user = user,
        tokens = tokens,
        loginMethod = loginMethod,
      )
    } else {
      // If tokens are not present, update the user of the current session, if there is one
      val currAuthState = authStateRepository.internalAuthState.value

      return if (currAuthState is InternalAuthState.Authenticated) {
        val updatedSession = currAuthState.session.copyAndUpdateUser(user)
        Result.success(updatedSession)
      } else {
        Result.failure(
          AuthenticationException("User must be authenticated before calling refresh user.")
        )
      }
    }
  }

  private fun createInternalAuthSession(
    user: InternalPrivyUser,
    tokens: AuthSessionResponseTokens,
    loginMethod: LoginMethod,
  ): InternalAuthSession {
    return InternalAuthSession(
      user = user,
      accessToken = tokens.token,
      refreshToken = tokens.refreshToken,
      identityToken = tokens.identityToken,
      loginMethod = loginMethod,
    )
  }

  public override fun <T> authenticatedOrDefault(
    onAuthenticated: (InternalAuthSession) -> T,
    default: () -> T
  ): T {
    // Current session
    val currAuthState = authStateRepository.internalAuthState.value

    return if (currAuthState is InternalAuthState.Authenticated) {
      onAuthenticated(currAuthState.session)
    } else {
      default()
    }
  }

  // Helper function to ensure user is authenticated before making wallet call
  // If authenticated: returns result.success with associated session
  // If unauthenticated: returns result.failure
  public override suspend fun <T> ensureAuthenticated(
    onAuthenticated: suspend (InternalAuthSession) -> Result<T>
  ): Result<T> {
    return refreshSessionIfNeeded()
      .fold(onSuccess = { onAuthenticated(it) }, onFailure = { Result.failure(it) })
  }

  override suspend fun refreshUser(): Result<InternalAuthSession> {
    // Current session
    val currAuthState = authStateRepository.internalAuthState.value

    return if (currAuthState is InternalAuthState.Authenticated) {
      currAuthState.session.refreshUser()
    } else {
      Result.failure(
        AuthenticationException("User must be authenticated before calling refresh user.")
      )
    }
  }

  override suspend fun refreshSession(): Result<InternalAuthSession> {
    // Current session
    val currAuthState = authStateRepository.internalAuthState.value

    return if (currAuthState is InternalAuthState.Authenticated) {
      currAuthState.session.refreshSession()
    } else {
      Result.failure(AuthenticationException("User must be authenticated before calling refresh."))
    }
  }

  override suspend fun logout() {
    authStateRepository.updateInternalAuthState(
      internalAuthState = InternalAuthState.Unauthenticated,
    )
  }

  override suspend fun refreshSessionIfNeeded(): Result<InternalAuthSession> {
    // Current session
    val currAuthState = authStateRepository.internalAuthState.value

    return if (currAuthState is InternalAuthState.Authenticated) {
      currAuthState.session.refreshSessionIfNeeded()
    } else {
      Result.failure(
        AuthenticationException("User must be authenticated before calling refresh session.")
      )
    }
  }

  private suspend fun updateAuthSession(authSession: InternalAuthSession) {
    // user is authenticated - update auth state
    // Note: external auth state should automatically update after setting internal state.
    authStateRepository.updateInternalAuthState(
      internalAuthState = InternalAuthState.Authenticated(session = authSession),
    )
  }

  private suspend fun InternalAuthSession.refreshSessionIfNeeded(): Result<InternalAuthSession> {
    val jwt = decodeJwt(this.accessToken)

    return if (jwt.isExpired()) {
      // If session is expired, try refreshing
      this.refreshSession()
    } else {
      // If session is already valid, return itself
      Result.success(this)
    }
  }

  // Helper function to refresh user in the scope of an existing session
  private suspend fun InternalAuthSession.refreshUser(): Result<InternalAuthSession> {
    val jwt = decodeJwt(this.accessToken)

    return if (jwt.isExpired()) {
      // If session is expired, refresh it, which will also refresh user
      this.refreshSession()
    } else {
      // If session is already valid, just refresh the user
      return authRepository
        .refreshUser(this.accessToken)
        .toResult()
        .fold(
          onSuccess = { refreshedUser ->
            val updatedSession = copyAndUpdateUser(refreshedUser)
            Result.success(updatedSession)
          },
          onFailure = { Result.failure(it) })
    }
  }

  /**
   * Updates the user object in the current auth session.
   *
   * @return The updated auth session
   */
  private suspend fun InternalAuthSession.copyAndUpdateUser(updatedUser: InternalPrivyUser): InternalAuthSession {
    val updatedSession = this.copy(user = updatedUser)
    updateAuthSession(updatedSession)
    return updatedSession
  }

  /**
   * Attempts to refresh the user session. If there is no network, this will immediately return
   * Result.failure, with no change to auth state. If there is network, a successful refresh will
   * update auth state to "Authenticated", and an unsuccessful refresh will update the auth state to
   * "Unauthenticated"
   */
  private suspend fun InternalAuthSession.refreshSession(): Result<InternalAuthSession> {
    // Important - if device isn't connected to internet, throw an error. If we attempt refresh call,
    // API will fail and we'll log the user out.
    val networkState = networkStateManager.current

    return networkState.failIfNetworkConfirmedDisconnectedElse {
      when (this.loginMethod) {
        LoginMethod.CustomAccessToken -> reAuthenticateWithCustomAuth()
        else -> {
          return refreshTokensWithPrivy(
            currentAccessToken = this.accessToken,
            currentRefreshToken = this.refreshToken,
            currentLoginMethod = this.loginMethod,
          )
        }
      }
    }
  }

  /**
   * Attempts to refresh the user session. If there is no network, this will immediately return
   * Result.failure, with no change to auth state. If there is network, a successful refresh will
   * update auth state to "Authenticated", and an unsuccessful refresh will update the auth state to
   * "Unauthenticated"
   */
  private suspend fun InternalAuthSessionTokens.refreshSession(): Result<InternalAuthSession> {
    val networkState = networkStateManager.current

    // Important - if device isn't connected to internet, throw an error. If we attempt refresh call,
    // API will fail and we'll log the user out.
    return networkState.failIfNetworkConfirmedDisconnectedElse {
      when (this.loginMethod) {
        LoginMethod.CustomAccessToken -> reAuthenticateWithCustomAuth()
        else -> {
          return refreshTokensWithPrivy(
            currentAccessToken = this.accessToken,
            currentRefreshToken = this.refreshToken,
            currentLoginMethod = this.loginMethod
          )
        }
      }
    }
  }

  private suspend fun refreshTokensWithPrivy(
    currentAccessToken: String,
    currentRefreshToken: String,
    currentLoginMethod: LoginMethod
  ): Result<InternalAuthSession> {
    // Using auth refresh service guarantees thread safety
    val refreshResult =
      authRefreshService
        .refreshSession(
          authToken = currentAccessToken,
          refreshToken = currentRefreshToken,
          currentLoginMethod = currentLoginMethod,
        )
        .toResult()

    return handleAuthResponse(authResult = refreshResult)
  }

  private suspend fun reAuthenticateWithCustomAuth(): Result<InternalAuthSession> {
    val customAuthAccessToken = tokenProvider?.invoke()

    if (customAuthAccessToken == null) {
      // since reauthenticating user failed, log them out
      privyLogger.debug("Custom access token from tokenProvider null. Logging user out.")
      this.logout()
      return Result.failure(
        AuthenticationException("Custom access token from tokenProvider null. Logging user out."),
      )
    }

    return this.internalLogin(
      loginType = LoginType.CustomAccessToken(token = customAuthAccessToken),
    )
  }
}