package io.privy.auth

import io.privy.analytics.ClientAnalyticsIdRepository
import io.privy.auth.customAuth.LoginWithCustomAuth
import io.privy.auth.email.LoginWithEmail
import io.privy.auth.internal.InternalAuthManager
import io.privy.auth.internal.InternalAuthSession
import io.privy.auth.internal.InternalAuthState
import io.privy.auth.internal.InternalAuthStateRepository
import io.privy.auth.oAuth.LoginWithOAuth
import io.privy.auth.passkey.LoginWithPasskey
import io.privy.auth.siwe.LoginWithSiwe
import io.privy.auth.sms.LoginWithSms
import io.privy.logging.PrivyLogger
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.flow.stateIn
import me.tatarka.inject.annotations.Inject

// This class serves as the main entry point into Privy auth for any module outside of
// the authentication module. This class should mostly direct request to the appropriate
// owner.
@Inject
public class RealPrivyAuth(
  private val internalAuthStateRepository: InternalAuthStateRepository,
  private val internalAuthManager: InternalAuthManager,
  private val clientAnalyticsIdRepository: ClientAnalyticsIdRepository,
  private val privyLogger: PrivyLogger,
  override val sms: LoginWithSms,
  override val email: LoginWithEmail,
  override val customAuth: LoginWithCustomAuth,
  override val siwe: LoginWithSiwe,
  override val oAuth: LoginWithOAuth,
  override val passkey: LoginWithPasskey
): PrivyAuth {
  private val scope = CoroutineScope(SupervisorJob() + Dispatchers.Main.immediate)

  // Consider SDK ready if auth state is anything other than NotReady
  override val isReady: Boolean
    get() = internalAuthStateRepository.currentAuthState !is InternalAuthState.NotReady

  // Only expose user object is there is an authenticated session
  override val user: PrivyUser?
    get() {
      val currentAuthState = this.internalAuthStateRepository.currentAuthState
      return if (currentAuthState is InternalAuthState.Authenticated) {
        // if there is an authenticated session return a user
        internalAuthManager.privyUser
      } else {
        // if no session, return null
        null
      }
    }

  // A flow that translates our internal auth state to external auth state
  // Subscribing to internal auth state updates, then mapping to external auth state,
  // is an async operations, so we should never rely on this flow having the latest value.
  // Instead, use internal auth state as source of truth.
  override val authStateUpdates: StateFlow<AuthState> =
    internalAuthStateRepository.internalAuthState
      .map { it.toAuthState() }
      .stateIn(
        scope = scope,
        started = SharingStarted.Eagerly,
        initialValue = AuthState.NotReady,
      )

  override suspend fun getAuthState(): AuthState {
    // Ensure initialization flow complete before returning the state
    internalAuthManager.awaitInitializationComplete();

    // Directly grab the value from internal auth state flow as it will be the most up to date
    return this.internalAuthStateRepository.currentAuthState.toAuthState();
  }

  override suspend fun onNetworkRestored() {
    // We only care about network restoration if we never determined auth state at SDK init
    if (internalAuthStateRepository.currentAuthState.hasBeenDetermined()) {
      privyLogger.debug(
        "Received network restored event, but auth state has already been determined. No op.")

      return
    }

    // By this point, we know auth state has not been determined, so we attempt to restore
    // the prior session
    internalAuthManager.restorePriorSessionIfNeeded()
  }

  override suspend fun hasPersistedAuthCredentials(): Boolean {
    return this.internalAuthManager.hasPersistedAuthCredentials()
  }

  override suspend fun logout() {
    clientAnalyticsIdRepository.clearClientAnalyticsId()
    internalAuthManager.logout()
  }

  override suspend fun awaitInitializationComplete() {
    return this.internalAuthManager.awaitInitializationComplete()
  }

  override suspend fun refreshSessionIfNeeded(): Result<InternalAuthSession> {
    return this.internalAuthManager.refreshSessionIfNeeded()
  }

  override suspend fun <T> ensureAuthenticated(onAuthenticated: suspend (InternalAuthSession) -> Result<T>): Result<T> {
    return this.internalAuthManager.ensureAuthenticated(onAuthenticated)
  }

  // Helper to convert internal auth state to external auth state
  private fun InternalAuthState.toAuthState(): AuthState {
    return when (this) {
      InternalAuthState.NotReady -> AuthState.NotReady
      InternalAuthState.AuthenticatedUnverified -> AuthState.AuthenticatedUnverified(Unit)
      // If authenticated, expose user object
      is InternalAuthState.Authenticated ->
        AuthState.Authenticated(user = internalAuthManager.privyUser)
      InternalAuthState.Unauthenticated -> {
        AuthState.Unauthenticated
      }
    }
  }
}