package io.privy.auth

import io.privy.auth.internal.InternalAuthManager
import io.privy.auth.internal.InternalLinkedAccount
import io.privy.auth.internal.embeddedEthereumWalletAccounts
import io.privy.auth.internal.embeddedSolanaWalletAccounts
import io.privy.auth.internal.primaryWalletForEntropyInfo
import io.privy.auth.internal.toEmbeddedWalletDetails
import io.privy.logging.PrivyLogger
import io.privy.wallet.EmbeddedWallet
import io.privy.wallet.EmbeddedWalletException
import io.privy.wallet.EmbeddedWalletManager
import io.privy.wallet.ethereum.EmbeddedEthereumWallet
import io.privy.wallet.ethereum.EmbeddedEthereumWalletFactory
import io.privy.wallet.solana.EmbeddedSolanaWallet
import io.privy.wallet.solana.EmbeddedSolanaWalletFactory
import me.tatarka.inject.annotations.Assisted
import me.tatarka.inject.annotations.Inject

/**
 * RealPrivyUser is simply a public facing abstraction of a user's auth session. This class should
 * be stateless and should dynamically derive values as they are requested. This ensures the return
 * values will be up to date. i.e. when user.linkedAccounts is accessed, it will dynamically
 * determine the users' linked accounts based on the latest auth session value
 */
@Inject
public class RealPrivyUser(
  @Assisted private val internalAuthManager: InternalAuthManager,
  private val embeddedEthereumWalletFactory: EmbeddedEthereumWalletFactory,
  private val embeddedSolanaWalletFactory: EmbeddedSolanaWalletFactory,
  private val embeddedWalletManager: EmbeddedWalletManager,
  private val privyLogger: PrivyLogger,
) : PrivyUser {
  // All properties on Privy user are derived, meaning they are determined at the time the value
  // is retrieved. This ensures that any updates to the internal session are taken into account and
  // all updates are reflected on the user object.
  override val id: String
    get() {
      return internalAuthManager.authenticatedOrDefault(
          onAuthenticated = { authSession -> authSession.user.id },
          default = {
            // If user is unauthenticated, return empty string for ID
            // This would only happen if app dev captured the user object when authenticated, and
            // then
            // user was logged out
            ""
          })
    }

  // Public facing LinkedAccounts
  override val linkedAccounts: List<LinkedAccount>
    get() {
      return internalLinkedAccounts.map { it.toLinkedAccount() }
    }

  // The identity token for this user, if configured in the Privy dashboard.
  override val identityToken: String?
    get() = internalAuthManager.authenticatedOrDefault({ it.identityToken }, { null })

  private val internalLinkedAccounts: List<InternalLinkedAccount>
    get() {
      return internalAuthManager.authenticatedOrDefault(
          onAuthenticated = { authSession -> authSession.user.linkedAccounts },
          default = {
            // If user us unauthenticated, return empty list
            // This would only happen if app dev captured the user object when authenticated, and
            // then
            // user was logged out
            emptyList()
          })
    }

  override val embeddedEthereumWallets: List<EmbeddedEthereumWallet>
    get() {
      val primaryEmbeddedWalletAccount = internalLinkedAccounts.primaryWalletForEntropyInfo()

      val embeddedWallets =
          primaryEmbeddedWalletAccount?.let { primaryWalletAccount ->
            internalLinkedAccounts.embeddedEthereumWalletAccounts().map {
              embeddedEthereumWalletFactory.create(
                  embeddedWalletDetails =
                      it.toEmbeddedWalletDetails(
                          entropyWallet = primaryEmbeddedWalletAccount))
            }
          }

      return embeddedWallets ?: emptyList()
    }

  override val embeddedSolanaWallets: List<EmbeddedSolanaWallet>
    get() {
      val primaryEmbeddedWalletAccount = internalLinkedAccounts.primaryWalletForEntropyInfo()

      val embeddedWallets =
          primaryEmbeddedWalletAccount?.let { primaryWalletAccount ->
            internalLinkedAccounts.embeddedSolanaWalletAccounts().map {
              embeddedSolanaWalletFactory.create(
                  embeddedWalletDetails =
                      it.toEmbeddedWalletDetails(
                          entropyWallet = primaryEmbeddedWalletAccount))
            }
          }

      return embeddedWallets ?: emptyList()
    }

  override suspend fun refresh(): Result<Unit> {
    return internalAuthManager.refreshUser().map {
      // refresh user returns InternalAuthSession, which we don't want to expose to user, so map
      // it to Unit. This way, developer can determine if refresh was successful
      Unit
    }
  }

  override suspend fun getAccessToken(): Result<String> {
    return internalAuthManager.refreshSessionIfNeeded().map { it.accessToken }
  }

  override suspend fun createEthereumWallet(allowAdditional: Boolean): Result<EmbeddedEthereumWallet> {
    return embeddedWalletManager.createEthereumWallet(allowAdditional = allowAdditional)
    .flatMap { newWalletAddress ->
      onWalletCreated(
        newWalletAddress = newWalletAddress, getWallets = { embeddedEthereumWallets },
      )
    }
  }

  override suspend fun createSolanaWallet(allowAdditional: Boolean): Result<EmbeddedSolanaWallet> {
    return embeddedWalletManager.createSolanaWallet(allowAdditional = allowAdditional)
    .flatMap { newWalletAddress ->
      onWalletCreated(
        newWalletAddress = newWalletAddress, getWallets = { embeddedSolanaWallets },
      )
    }
  }

  /**
   * A chain agnostic, reusable function that force refreshes auth session when new wallet is
   * created, which will retrieve the latest linked accounts, including the newly created wallet.
   *
   * @param newWalletAddress The address of the newly created wallet
   * @param getWallets A lambda that allows the caller to specify the list of wallets to search
   *   from. This allows the caller to specify the EmbeddedWallet type
   */
  private suspend fun <T : EmbeddedWallet> onWalletCreated(
    newWalletAddress: String,
    getWallets: () -> List<T>
  ): Result<T> {
    privyLogger.info("New embedded wallet created. Wallet address: $newWalletAddress")

    return internalAuthManager.refreshSession().flatMap {
      // Get newly created embedded wallet from list
      // important that we call getWallets() after refreshing the session
      val newEmbeddedWallet = getWallets().firstOrNull { it.address == newWalletAddress }

      return if (newEmbeddedWallet == null) {
        // Shouldn't happen
        Result.failure(
          exception = EmbeddedWalletException(message = "Failed to create embedded wallet.")
        )
      } else {
        // Once session is refreshed, try connecting wallet
        embeddedWalletManager.connectWalletIfNeeded()

        // return embedded wallet
        Result.success(newEmbeddedWallet)
      }
    }
  }

  override fun toString(): String {
    val defaultToString = super.toString()

    return """ 
    PrivyUser(
      ID: '$id', 
      
      Linked Accounts:
      ${linkedAccounts.map { it.prettyPrint() }}, 
      
      Instance: ${defaultToString}, 
      
      Identity Token: ${identityToken?.take(10)?.let { "$it..." } ?: "None"}
    )
    """
        .trimIndent()
  }

  private fun LinkedAccount.prettyPrint(): String {
    return when (this) {
      is LinkedAccount.CustomAuth ->
          """
        
        CustomAuth(
          User id: ${this.customUserId} 
        )
         
      """
              .trimIndent()
      is LinkedAccount.GoogleOAuthAccount ->
          """
        
        GoogleOAuth(
          Subject: ${this.subject}
          Email: ${this.email}
          Name: ${this.name ?: "N/A"}
          First Verified At: ${this.firstVerifiedAt}
          Latest Verified At: ${this.latestVerifiedAt}
        )
         
      """
              .trimIndent()
      is LinkedAccount.TwitterOAuthAccount ->
          """
        
        TwitterOAuth(
          Subject: ${this.subject}
          Username: ${this.username}
          Name: ${this.name ?: "N/A"}
          Email: ${this.email ?: "N/A"}
          Profile Picture URL: ${this.profilePictureUrl ?: "N/A"}
          First Verified At: ${this.firstVerifiedAt}
          Latest Verified At: ${this.latestVerifiedAt}
        )
         
      """
              .trimIndent()
      is LinkedAccount.DiscordOAuthAccount ->
          """
        
        DiscordOAuth(
          Subject: ${this.subject}
          Username: ${this.username}
          Email: ${this.email ?: "N/A"}
          First Verified At: ${this.firstVerifiedAt}
          Latest Verified At: ${this.latestVerifiedAt}
        )
         
      """
              .trimIndent()
      is LinkedAccount.EmailAccount ->
          """
        
        Email(
          address: ${this.emailAddress} 
        )
         
      """
              .trimIndent()
      is LinkedAccount.EmbeddedEthereumWalletAccount ->
          """
        
        EthereumWallet(
          id=$id,
          address=$address,
          hdWalletIndex=$hdWalletIndex,
          chainId=$chainId,
          recoveryMethod=$recoveryMethod,
        )
        
      """
              .trimIndent()
      is LinkedAccount.EmbeddedSolanaWalletAccount ->
          """
        
        SolanaWallet(
          id=$id,
          address=$address,
          hdWalletIndex=$hdWalletIndex,
          chainId=$chainId,
          recoveryMethod=$recoveryMethod,
        )
        
      """
              .trimIndent()
      is LinkedAccount.ExternalWalletAccount ->
          """
        
        ExternalWallet(
          address=$address,
          chainId=$chainId,
          chainType=${chainType.name},
          recoveryMethod=$connectorType
        )
        
      """
              .trimIndent()
      is LinkedAccount.PhoneAccount ->
          """

        SMS(
          phoneNumber: ${this.phoneNumber},
        )

      """
              .trimIndent()
      is LinkedAccount.PasskeyAccount ->
          """

        Passkey(
          Credential ID: ${this.credentialId}
          Authenticator Name: ${this.authenticatorName ?: "N/A"}
          Created With Browser: ${this.createdWithBrowser ?: "N/A"}
          Created With OS: ${this.createdWithOs ?: "N/A"}
          Created With Device: ${this.createdWithDevice ?: "N/A"}
          Public Key: ${this.publicKey ?: "N/A"}
          Enrolled In MFA: ${this.enrolledInMfa}
          Verified At: ${this.verifiedAt}
          First Verified At: ${this.firstVerifiedAt ?: "N/A"}
          Latest Verified At: ${this.latestVerifiedAt ?: "N/A"}
        )

      """
              .trimIndent()
    }
  }

  override fun equals(other: Any?): Boolean {
    if (other !is RealPrivyUser) {
      return false
    }

    // A user is equal to another user if they have the same ID and same set of
    // linked accounts.
    return id == other.id && linkedAccounts == other.linkedAccounts
  }

  override fun hashCode(): Int {
    var result = id.hashCode()
    result = 31 * result + linkedAccounts.hashCode()
    return result
  }
}
