package io.privy.auth.passkey

import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable

// ============================================================================
// API Response Models - FROM Privy API (snake_case)
// ============================================================================

/** Wrapper for registration options API response */
@Serializable
internal data class ApiRegistrationOptionsResponse(
    @SerialName("options") val options: ApiRegistrationOptions
)

/** Registration options from Privy API */
@Serializable
internal data class ApiRegistrationOptions(
    @SerialName("challenge") val challenge: String,
    @SerialName("rp") val rp: ApiRelyingPartyInfo,
    @SerialName("user") val user: ApiUserInfo,
    @SerialName("pub_key_cred_params") val pubKeyCredParams: List<ApiPubKeyCredParam>,
    @SerialName("timeout") val timeout: Long? = 60000,
    @SerialName("authenticator_selection") val authenticatorSelection: ApiAuthenticatorSelection? = null,
    @SerialName("exclude_credentials") val excludeCredentials: List<ApiCredentialDescriptor>? = null,
    @SerialName("attestation") val attestation: String? = null,
    @SerialName("extensions") val extensions: Map<String, kotlinx.serialization.json.JsonElement>? = null,
)

/** Wrapper for authentication options API response */
@Serializable
internal data class ApiAuthenticationOptionsResponse(
    @SerialName("options") val options: ApiAuthenticationOptions
)

/** Authentication options from Privy API */
@Serializable
internal data class ApiAuthenticationOptions(
    @SerialName("challenge") val challenge: String,
    @SerialName("rp_id") val rpId: String,
    @SerialName("timeout") val timeout: Long? = 60000,
    @SerialName("user_verification") val userVerification: String? = "preferred",
    @SerialName("allow_credentials") val allowCredentials: List<ApiCredentialDescriptor>? = null,
)

@Serializable
internal data class ApiRelyingPartyInfo(
    @SerialName("id") val id: String,
    @SerialName("name") val name: String,
)

@Serializable
internal data class ApiUserInfo(
    @SerialName("id") val id: String,
    @SerialName("name") val name: String,
    @SerialName("display_name") val displayName: String,
)

@Serializable
internal data class ApiPubKeyCredParam(
    @SerialName("type") val type: String,
    @SerialName("alg") val alg: Int,
)

@Serializable
internal data class ApiAuthenticatorSelection(
    @SerialName("authenticator_attachment") val authenticatorAttachment: String? = null,
    @SerialName("require_resident_key") val requireResidentKey: Boolean? = null,
    @SerialName("resident_key") val residentKey: String? = null,
    @SerialName("user_verification") val userVerification: String? = null,
)

@Serializable
internal data class ApiCredentialDescriptor(
    @SerialName("type") val type: String,
    @SerialName("id") val id: String,
    @SerialName("transports") val transports: List<String>? = null,
)

// ============================================================================
// API Request Models - TO Privy API (snake_case)
// ============================================================================

/** Client extension results for WebAuthn operations */
@Serializable
internal data class ApiClientExtensionResults(
    @SerialName("app_id") val appId: String? = null,
    @SerialName("cred_props") val credProps: String? = null,
    @SerialName("hmac_create_secret") val hmacCreateSecret: String? = null,
)

/** Passkey registration response in API format */
@Serializable
internal data class ApiPasskeyRegistrationResponse(
    @SerialName("type") val type: String,
    @SerialName("id") val id: String,
    @SerialName("raw_id") val rawId: String,
    @SerialName("response") val response: ApiAuthenticatorAttestationResponse,
    @SerialName("authenticator_attachment") val authenticatorAttachment: String? = null,
    @SerialName("client_extension_results") val clientExtensionResults: ApiClientExtensionResults,
)

@Serializable
internal data class ApiAuthenticatorAttestationResponse(
    @SerialName("client_data_json") val clientDataJSON: String,
    @SerialName("attestation_object") val attestationObject: String,
    @SerialName("transports") val transports: List<String>? = null,
    @SerialName("public_key_algorithm") val publicKeyAlgorithm: Int? = null,
    @SerialName("public_key") val publicKey: String? = null,
    @SerialName("authenticator_data") val authenticatorData: String? = null,
)

/** Passkey authentication response in API format */
@Serializable
internal data class ApiPasskeyAuthenticationResponse(
    @SerialName("type") val type: String,
    @SerialName("id") val id: String,
    @SerialName("raw_id") val rawId: String,
    @SerialName("response") val response: ApiAuthenticatorAssertionResponse,
    @SerialName("authenticator_attachment") val authenticatorAttachment: String? = null,
    @SerialName("client_extension_results") val clientExtensionResults: ApiClientExtensionResults,
)

@Serializable
internal data class ApiAuthenticatorAssertionResponse(
    @SerialName("client_data_json") val clientDataJSON: String,
    @SerialName("authenticator_data") val authenticatorData: String,
    @SerialName("signature") val signature: String,
    @SerialName("user_handle") val userHandle: String? = null,
)

/** Request body for passkey registration */
@Serializable
internal data class PasskeyRegisterRequest(
    @SerialName("authenticator_response") val authenticatorResponse: ApiPasskeyRegistrationResponse,
    @SerialName("relying_party") val relyingParty: String,
)

/** Request body for passkey authentication */
@Serializable
internal data class PasskeyAuthenticateRequest(
    @SerialName("authenticator_response") val authenticatorResponse: ApiPasskeyAuthenticationResponse,
    @SerialName("challenge") val challenge: String,
    @SerialName("relying_party") val relyingParty: String,
)

// ============================================================================
// Native Models - For Android Credential Manager (camelCase)
// ============================================================================

/** Registration options for Android Credential Manager */
@Serializable
public data class RegistrationOptions(
    val challenge: String,
    val rp: RelyingPartyInfo,
    val user: UserInfo,
    val pubKeyCredParams: List<PubKeyCredParam>,
    val timeout: Long? = 60000,
    val authenticatorSelection: AuthenticatorSelection? = null,
    val excludeCredentials: List<CredentialDescriptor>? = null,
    val attestation: String? = null,
)

/** Authentication options for Android Credential Manager */
@Serializable
public data class AuthenticationOptions(
    val challenge: String,
    val rpId: String,
    val timeout: Long? = 60000,
    val userVerification: String? = "preferred",
    val allowCredentials: List<CredentialDescriptor>? = null,
)

@Serializable
public data class RelyingPartyInfo(
    val id: String,
    val name: String,
)

@Serializable
public data class UserInfo(
    val id: String,
    val name: String,
    val displayName: String,
)

@Serializable
public data class PubKeyCredParam(
    val type: String,
    val alg: Int,
)

@Serializable
public data class AuthenticatorSelection(
    val authenticatorAttachment: String? = null,
    val requireResidentKey: Boolean? = null,
    val residentKey: String? = null,
    val userVerification: String? = null,
)

@Serializable
public data class CredentialDescriptor(
    val type: String,
    val id: String,
    val transports: List<String>? = null,
)

/** Passkey registration response from Android Credential Manager */
@Serializable
public data class PasskeyRegistrationResponse(
    val type: String,
    val id: String,
    val rawId: String,
    val response: AuthenticatorAttestationResponse,
    val authenticatorAttachment: String? = null,
    val clientExtensionResults: Map<String, String> = emptyMap(),
)

@Serializable
public data class AuthenticatorAttestationResponse(
    val clientDataJSON: String,
    val attestationObject: String,
    val transports: List<String>? = null,
    val publicKeyAlgorithm: Int? = null,
    val publicKey: String? = null,
    val authenticatorData: String? = null,
)

/** Passkey authentication response from Android Credential Manager */
@Serializable
public data class PasskeyAuthenticationResponse(
    val type: String,
    val id: String,
    val rawId: String,
    val response: AuthenticatorAssertionResponse,
    val authenticatorAttachment: String? = null,
    val clientExtensionResults: Map<String, String> = emptyMap(),
)

@Serializable
public data class AuthenticatorAssertionResponse(
    val clientDataJSON: String,
    val authenticatorData: String,
    val signature: String,
    val userHandle: String? = null,
)

// ============================================================================
// Conversion Functions - API to Native
// ============================================================================

/** Convert API registration options to native format for Android Credential Manager */
internal fun ApiRegistrationOptions.toNative(): RegistrationOptions = RegistrationOptions(
    challenge = challenge,
    rp = RelyingPartyInfo(id = rp.id, name = rp.name),
    user = UserInfo(id = user.id, name = user.name, displayName = user.displayName),
    pubKeyCredParams = pubKeyCredParams.map { PubKeyCredParam(type = it.type, alg = it.alg) },
    timeout = timeout,
    authenticatorSelection = authenticatorSelection?.let {
        AuthenticatorSelection(
            authenticatorAttachment = it.authenticatorAttachment,
            requireResidentKey = it.requireResidentKey,
            residentKey = it.residentKey,
            userVerification = it.userVerification,
        )
    },
    excludeCredentials = excludeCredentials?.map { CredentialDescriptor(id = it.id, type = it.type, transports = it.transports) },
    attestation = attestation,
)

/** Convert API authentication options to native format for Android Credential Manager */
internal fun ApiAuthenticationOptions.toNative(): AuthenticationOptions = AuthenticationOptions(
    challenge = challenge,
    rpId = rpId,
    timeout = timeout,
    userVerification = userVerification,
    allowCredentials = allowCredentials?.map { CredentialDescriptor(id = it.id, type = it.type, transports = it.transports) },
)

// ============================================================================
// Conversion Functions - Native to API
// ============================================================================

/** Convert native PasskeyRegistrationResponse to API format */
internal fun PasskeyRegistrationResponse.toApiFormat(): ApiPasskeyRegistrationResponse = ApiPasskeyRegistrationResponse(
    type = type,
    id = id,
    rawId = rawId,
    response = ApiAuthenticatorAttestationResponse(
        clientDataJSON = response.clientDataJSON,
        attestationObject = response.attestationObject,
        transports = response.transports,
        publicKeyAlgorithm = response.publicKeyAlgorithm,
        publicKey = response.publicKey,
        authenticatorData = response.authenticatorData,
    ),
    authenticatorAttachment = authenticatorAttachment,
    clientExtensionResults = ApiClientExtensionResults(
        appId = clientExtensionResults["appid"],
        credProps = clientExtensionResults["credProps"],
        hmacCreateSecret = clientExtensionResults["hmacCreateSecret"],
    ),
)

/** Convert native PasskeyAuthenticationResponse to API format */
internal fun PasskeyAuthenticationResponse.toApiFormat(): ApiPasskeyAuthenticationResponse = ApiPasskeyAuthenticationResponse(
    type = type,
    id = id,
    rawId = rawId,
    response = ApiAuthenticatorAssertionResponse(
        clientDataJSON = response.clientDataJSON,
        authenticatorData = response.authenticatorData,
        signature = response.signature,
        userHandle = response.userHandle,
    ),
    authenticatorAttachment = authenticatorAttachment,
    clientExtensionResults = ApiClientExtensionResults(
        appId = clientExtensionResults["appid"],
        credProps = clientExtensionResults["credProps"],
        hmacCreateSecret = clientExtensionResults["hmacCreateSecret"],
    ),
)
