package io.privy.auth.passkey

import io.privy.auth.AuthRepository
import io.privy.auth.LoginType
import io.privy.auth.PrivyUser
import io.privy.auth.internal.InternalAuthManager
import io.privy.logging.PrivyLogger
import io.privy.network.toResult
import me.tatarka.inject.annotations.Inject

/**
 * Implementation of LoginWithPasskey that handles passkey authentication flows.
 *
 * Coordinates between:
 * - AuthRepository: For server API calls
 * - PasskeyManager: For platform-specific credential operations
 * - InternalAuthManager: For session management
 */
@Inject
public class RealLoginWithPasskey(
    private val authRepository: AuthRepository,
    private val internalAuthManager: InternalAuthManager,
    private val passkeyManager: PasskeyManager,
    private val privyLogger: PrivyLogger,
) : LoginWithPasskey {

  override suspend fun signup(
      relyingParty: String,
      displayName: String?,
  ): Result<PrivyUser> {
    // 1. Generate challenge from server
    privyLogger.debug("PasskeyAuth: Generating signup challenge")

    val options = authRepository.generatePasskeySignupOptions(relyingParty).toResult().getOrElse { error ->
      privyLogger.error("PasskeyAuth: Failed to generate signup options - ${error.message}")
      return Result.failure(error)
    }

    // Override display name if provided
    val finalOptions = displayName?.let { options.copy(user = options.user.copy(displayName = it)) } ?: options

    // Match Privy API specification for passkey authentication
    val platformOptions = finalOptions.copy(
        authenticatorSelection = AuthenticatorSelection(
            authenticatorAttachment = null,  // API doesn't specify - allow any authenticator type
            residentKey = "required",  // API requires discoverable credentials
            userVerification = "required"  // API requires biometric/PIN verification
        )
    )

    // 2. Create passkey with platform manager
    privyLogger.debug("PasskeyAuth: Creating passkey credential")

    val credential = passkeyManager.createPasskey(platformOptions).getOrElse { error ->
      privyLogger.error("PasskeyAuth: Failed to create passkey - ${error.message}")
      return Result.failure(error)
    }

    // 3. Submit to server and authenticate
    privyLogger.debug("PasskeyAuth: Submitting credential to server")
    privyLogger.debug("PasskeyAuth: Credential = $credential")

    val loginType = LoginType.PasskeyRegistration(
        credential = credential,
        relyingParty = relyingParty
    )

    return internalAuthManager.login(loginType = loginType).onSuccess { user ->
      privyLogger.debug("PasskeyAuth: Signup successful for user ${user.id}")
    }.onFailure { error ->
      privyLogger.error("PasskeyAuth: Signup failed - ${error.message}")
    }
  }

  override suspend fun login(relyingParty: String): Result<PrivyUser> {
    // 1. Generate challenge from server
    privyLogger.debug("PasskeyAuth: Generating login challenge")

    val options = authRepository.generatePasskeyAuthenticationOptions(relyingParty).toResult().getOrElse { error ->
      privyLogger.error("PasskeyAuth: Failed to generate login options - ${error.message}")
      return Result.failure(error)
    }

    // 2. Get passkey with platform manager
    privyLogger.debug("PasskeyAuth: Retrieving passkey credential")

    val credential = passkeyManager.getPasskey(options).getOrElse { error ->
      privyLogger.error("PasskeyAuth: Failed to retrieve passkey - ${error.message}")
      return Result.failure(error)
    }

    // 3. Submit to server and authenticate
    privyLogger.debug("PasskeyAuth: Submitting credential to server")

    val loginType = LoginType.PasskeyAuthentication(
        credential = credential,
        challenge = options.challenge,
        relyingParty = relyingParty,
    )

    return internalAuthManager.login(loginType = loginType).onSuccess { user ->
      privyLogger.debug("PasskeyAuth: Login successful for user ${user.id}")
    }.onFailure { error ->
      privyLogger.error("PasskeyAuth: Login failed - ${error.message}")
    }
  }
}
