package io.privy.auth.session.internal

import io.privy.auth.internal.InternalLinkedAccount
import io.privy.auth.internal.InternalPrivyUser
import io.privy.auth.session.LinkedAccountResponse
import io.privy.auth.session.PrivyUserResponse
import io.privy.wallet.ChainType

internal fun PrivyUserResponse.toInternalPrivyUser(): InternalPrivyUser {
  return with(this) {
    InternalPrivyUser(
        id = id,
      linkedAccounts = linkedAccounts.mapNotNull { it.toInternalLinkedAccount() },
      )
  }
}

private fun LinkedAccountResponse.toInternalLinkedAccount(): InternalLinkedAccount? {
  return when (this) {
    is LinkedAccountResponse.CustomAuth ->
        InternalLinkedAccount.CustomAuth(
            customUserId = customUserId,
            firstVerifiedAt = firstVerifiedAt,
            latestVerifiedAt = latestVerifiedAt,
        )
    is LinkedAccountResponse.WalletAccount ->
        if (this.isPrivyWallet()) {
          this.toInternalEmbeddedWalletOrNull()
        } else {
          this.toExternalWalletAccount()
        }
    is LinkedAccountResponse.PhoneNumberAccount ->
        InternalLinkedAccount.PhoneAccount(phoneNumber = phoneNumber)
    is LinkedAccountResponse.EmailAddressAccount ->
        InternalLinkedAccount.EmailAccount(emailAddress = emailAddress)
    is LinkedAccountResponse.GoogleOAuthAccount ->
        InternalLinkedAccount.GoogleOAuthAccount(
            subject = subject,
            email = email,
            name = name,
            firstVerifiedAt = firstVerifiedAt,
            latestVerifiedAt = latestVerifiedAt
        )
    is LinkedAccountResponse.TwitterOAuthAccount ->
        InternalLinkedAccount.TwitterOAuthAccount(
            subject = subject,
            username = username,
            name = name,
            email = email,
            profilePictureUrl = profilePictureUrl,
            firstVerifiedAt = firstVerifiedAt,
            latestVerifiedAt = latestVerifiedAt
        )
    is LinkedAccountResponse.DiscordOAuthAccount ->
        InternalLinkedAccount.DiscordOAuthAccount(
            subject = subject,
            username = username,
            email = email,
            firstVerifiedAt = firstVerifiedAt,
            latestVerifiedAt = latestVerifiedAt
        )
    is LinkedAccountResponse.PasskeyAccount ->
        InternalLinkedAccount.PasskeyAccount(
            credentialId = credentialId,
            authenticatorName = authenticatorName,
            createdWithBrowser = createdWithBrowser,
            createdWithOs = createdWithOs,
            createdWithDevice = createdWithDevice,
            publicKey = publicKey,
            enrolledInMfa = enrolledInMfa,
            verifiedAt = verifiedAt,
            firstVerifiedAt = firstVerifiedAt,
            latestVerifiedAt = latestVerifiedAt
        )
    is LinkedAccountResponse.UnknownAccount -> null
  }
}
private fun LinkedAccountResponse.WalletAccount.toExternalWalletAccount(): InternalLinkedAccount.ExternalWalletAccount {
    return InternalLinkedAccount.ExternalWalletAccount(
        address = address,
        chainType = chainTypeConverter(chainType),
        chainId = chainId,
        walletClientType = walletClientType,
        connectorType = connectorType,
        firstVerifiedAt = firstVerifiedAt,
        latestVerifiedAt = latestVerifiedAt
    )
}

private fun LinkedAccountResponse.WalletAccount.toInternalEmbeddedWalletOrNull():
    InternalLinkedAccount.EmbeddedWalletAccount? {
  if (hdWalletIndex == null) {
    // privy embedded wallet should always have hd wallet index
    return null
  }

  return if (chainType == "solana") {
    InternalLinkedAccount.EmbeddedSolanaWalletAccount(
        id = id,
        address = address,
        chainId = chainId,
        recoveryMethod = recoveryMethod,
        hdWalletIndex = hdWalletIndex,
    )
  } else {
    InternalLinkedAccount.EmbeddedEthereumWalletAccount(
        id = id,
        address = address,
        chainId = chainId,
        recoveryMethod = recoveryMethod,
        hdWalletIndex = hdWalletIndex,
        firstVerifiedAt = firstVerifiedAt,
        latestVerifiedAt = latestVerifiedAt,
      )
  }
}

private fun chainTypeConverter(chainType: String?): ChainType {
   return when (chainType?.lowercase()) {
        "ethereum" -> ChainType.Ethereum
        "solana" -> ChainType.Solana
        else -> ChainType.Ethereum // Default to Ethereum if null or unknown
    }
}
