package io.privy.auth

import io.privy.wallet.ChainType

/**
 * A sealed type representing an account linked to a [PrivyUser]. Linked accounts can either be an
 * authentication method, or an embedded wallet. See all options below.
 */
public sealed interface LinkedAccount {
  /**
   * The linked account that represents when a user authenticates with a custom authentication
   * method,
   */
  public data class CustomAuth(
      val customUserId: String,
      val firstVerifiedAt: Int?,
      val latestVerifiedAt: Int?
  ) : LinkedAccount

  /**
   * A base type that represents an embedded wallet. Concrete types are
   * [EmbeddedEthereumWalletAccount] and [EmbeddedSolanaWalletAccount]
   */
  public sealed interface EmbeddedWalletAccount : LinkedAccount {
    public val address: String
    public val chainId: String?
    public val recoveryMethod: String?
    public val hdWalletIndex: Int
  }

  /** The linked account that represents a user's embedded Ethereum wallet */
  public data class EmbeddedEthereumWalletAccount(
      override val address: String,
      override val chainId: String?,
      override val recoveryMethod: String?,
      override val hdWalletIndex: Int
  ) : EmbeddedWalletAccount

  /** The linked account that represents a user's embedded Solana wallet */
  public data class EmbeddedSolanaWalletAccount(
      override val address: String,
      override val chainId: String?,
      override val recoveryMethod: String?,
      override val hdWalletIndex: Int
  ) : EmbeddedWalletAccount

    public data class ExternalWalletAccount(
        val address: String,
        val chainType: ChainType,
        val chainId: String?,
        val walletClientType: String?,
        val connectorType: String?,
        val firstVerifiedAt: Int?,
        val latestVerifiedAt: Int?
    ) : LinkedAccount

  /** The linked account that represents when a user authenticates via SMS */
  public data class PhoneAccount(val phoneNumber: String) : LinkedAccount

  /** The linked account that represents when a user authenticates via email */
  public data class EmailAccount(val emailAddress: String) : LinkedAccount

  /** The linked account that represents when a user authenticates via Google OAuth */
  public data class GoogleOAuthAccount(
      val subject: String,
      val email: String,
      val name: String?,
      val firstVerifiedAt: Int?,
      val latestVerifiedAt: Int?
  ) : LinkedAccount

  /** The linked account that represents when a user authenticates via Twitter OAuth */
  public data class TwitterOAuthAccount(
      val subject: String,
      val username: String,
      val name: String?,
      val email: String?,
      val profilePictureUrl: String?,
      val firstVerifiedAt: Int?,
      val latestVerifiedAt: Int?
  ) : LinkedAccount

  /** The linked account that represents when a user authenticates via Discord OAuth */
  public data class DiscordOAuthAccount(
      val subject: String,
      val username: String,
      val email: String?,
      val firstVerifiedAt: Int?,
      val latestVerifiedAt: Int?
  ) : LinkedAccount
}

// Ethereum Embedded wallet with hdWalletIndex == 0 is the primary wallet
public fun List<LinkedAccount>.primaryEmbeddedEthereumWalletAccountOrNull():
    LinkedAccount.EmbeddedEthereumWalletAccount? {
  return this.filterIsInstance<LinkedAccount.EmbeddedEthereumWalletAccount>().firstOrNull {
    it.hdWalletIndex == 0
  }
}

// Solana embedded wallet with hdWalletIndex == 0 is the primary wallet
public fun List<LinkedAccount>.primaryEmbeddedSolanaWalletAccountOrNull():
    LinkedAccount.EmbeddedSolanaWalletAccount? {
  return this.filterIsInstance<LinkedAccount.EmbeddedSolanaWalletAccount>().firstOrNull {
    it.hdWalletIndex == 0
  }
}
