package io.privy.auth

import io.privy.auth.customAuth.TokenProvider
import io.privy.auth.internal.InternalAuthSession
import io.privy.auth.internal.InternalSiweMessage
import io.privy.auth.oAuth.OAuthInitRequest
import io.privy.auth.oAuth.OAuthInitResponse
import io.privy.auth.otp.OtpRequestType

public interface AuthManager {
  public val privyUser: PrivyUser

  // A state flow that can be used to subscribe to auth state updates
  // public val authState: StateFlow<AuthState>

  // public val internalAuthState: StateFlow<InternalAuthState>

  // For custom auth refresh flow
  public fun setTokenProvider(tokenProvider: TokenProvider)

  /**
   * Sends OTP to the requested destination
   *
   * @param otpRequestType Either email or SMS
   * @return Result type with no associated value representing if OTP was sent successfully or not
   */
  public suspend fun sendOtp(otpRequestType: OtpRequestType): Result<Unit>

  public suspend fun login(loginType: LoginType): Result<PrivyUser>

  // Helper function to return a value if user is authenticated,
  // or else return the provided default value
  public fun <T> authenticatedOrDefault(
      onAuthenticated: (InternalAuthSession) -> T,
      default: () -> T
  ): T

  // Helper function to ensure user is authenticated before making wallet call
  // If authenticated: returns result.success with associated session
  // If unauthenticated: returns result.failure
  public suspend fun <T> ensureAuthenticated(
      onAuthenticated: suspend (InternalAuthSession) -> Result<T>
  ): Result<T>

  /**
   * Force refreshes existing session.
   *
   * @return Result.success if session was successfully refreshed or Result.failure if there is no current
   * session to refresh or if refresh fails
   */
  public suspend fun refreshSession(): Result<InternalAuthSession>

  public suspend fun linkAccount(loginType: LoginType): Result<Unit>

  /**
   * Checks if the current session is still valid. If the session is expired, it attempts to refresh
   * it.
   *
   * @return Result.success with a valid, refreshed session if needed, or the existing session if
   *   it's still valid. Result.failure if there is no current session to refresh or if the refresh
   *   fails.
   */
  public suspend fun refreshSessionIfNeeded(): Result<InternalAuthSession>

  /**
   * Refreshes the current session's user. If this session is expired, the session will be
   * refreshed.
   *
   * @return Result.success with a valid session, containing the updated user Result.failure if
   *   there is no current session to refresh or if refresh fails
   */
  public suspend fun refreshUser(): Result<InternalAuthSession>

  /**
   * Generates a SIWE (Sign-In with Ethereum) message for the given wallet address.
   *
   * @param walletAddress The Ethereum wallet address for which to generate the SIWE message
   * @return Result.success with a GenerateSiweMessageResponse containing the nonce and other
   *   details needed to construct the SIWE message, or Result.failure if the generation fails
   */
  public suspend fun generateSiweMessage(walletAddress: String): Result<InternalSiweMessage>

  /**
   * Returns the access token if the user is authenticated If the current access token is expired,
   * this will attempt to refresh the session, and return the updated access token.
   *
   * If the user is not authenticated, will return null
   */
  // TODO: impl this
  // public suspend fun getAccessToken(): String?

  public suspend fun logout()

  public suspend fun generateOAuthUrl(oAuthInitRequest: OAuthInitRequest): Result<OAuthInitResponse>
}
