package io.privy.auth.internal

import io.privy.auth.LinkedAccount
import io.privy.wallet.ChainType
import io.privy.wallet.EmbeddedWalletDetails
import io.privy.wallet.PrivyStack

public data class InternalAuthSession(
    val user: InternalPrivyUser,
    val accessToken: String,
    val refreshToken: String,
    val identityToken: String?,
    val loginMethod: LoginMethod
)

public data class InternalPrivyUser(
    val id: String,
    val linkedAccounts: List<InternalLinkedAccount>,
) {
  public fun hasEmbeddedWallet(): Boolean {
    val embeddedWallet = linkedAccounts.firstOrNull { it.isEmbeddedWallet() }
    return embeddedWallet != null
  }

  public val emailAddress: String?
    get() {
      return linkedAccounts
        .filterIsInstance<InternalLinkedAccount.EmailAccount>()
        .firstOrNull()
        ?.emailAddress
    }

  public val phoneNumber: String?
    get() {
      return linkedAccounts
          .filterIsInstance<InternalLinkedAccount.PhoneAccount>()
          .firstOrNull()
          ?.phoneNumber
    }
}

/**
 * Internal domain entities for LinkedAccounts. Will almost be 1:1 mapping with external
 * LinkedAccounts, but good to have separation between public entities and storage entities
 */
public sealed interface InternalLinkedAccount {
  public fun toLinkedAccount(): LinkedAccount

  public data class CustomAuth(
      val customUserId: String,
      // TODO: UPDATE THESE TO DATE TYPE
      val firstVerifiedAt: Int?,
      val latestVerifiedAt: Int?
  ) : InternalLinkedAccount {
    override fun toLinkedAccount(): LinkedAccount =
        LinkedAccount.CustomAuth(
            customUserId = customUserId,
            firstVerifiedAt = firstVerifiedAt,
            latestVerifiedAt = latestVerifiedAt,
        )
  }

  public sealed interface EmbeddedWalletAccount : InternalLinkedAccount {
    // The id of the wallet in the wallets api
    public val id: String?
    public val address: String
    public val chainId: String?
    public val recoveryMethod: String?
    public val hdWalletIndex: Int
  }

  public data class EmbeddedEthereumWalletAccount(
      // The id of the wallet in the wallets api
      override val id: String?,
      override val address: String,
      override val chainId: String?,
      // "privy" or "privy-v2": privy escrow of the recovery material
      override val recoveryMethod: String?,
      override val hdWalletIndex: Int,
      val firstVerifiedAt: Int?,
      val latestVerifiedAt: Int?,
  ) : EmbeddedWalletAccount {
    override fun toLinkedAccount(): LinkedAccount =
        LinkedAccount.EmbeddedEthereumWalletAccount(
            id = id,
            address = address,
            chainId = chainId,
            recoveryMethod = recoveryMethod,
            hdWalletIndex = hdWalletIndex,
        )
  }

  public data class EmbeddedSolanaWalletAccount(
      // The id of the wallet in the wallets api
      override val id: String?,
      override val address: String,
      override val chainId: String?,
      // "privy" or "privy-v2": privy escrow of the recovery material
      override val recoveryMethod: String?,
      override val hdWalletIndex: Int,
  ) : EmbeddedWalletAccount {
    override fun toLinkedAccount(): LinkedAccount =
        LinkedAccount.EmbeddedSolanaWalletAccount(
            id = id,
            address = address,
            chainId = chainId,
            recoveryMethod = recoveryMethod,
            hdWalletIndex = hdWalletIndex,
        )
  }

  public data class ExternalWalletAccount(
      val address: String,
      val chainType: ChainType,
      val chainId: String?,
      val walletClientType: String?,
      val connectorType: String?,
      val firstVerifiedAt: Int?,
      val latestVerifiedAt: Int?
  ) : InternalLinkedAccount {
    override fun toLinkedAccount(): LinkedAccount =
        LinkedAccount.ExternalWalletAccount(
            address = address,
            chainType = chainType,
            chainId = chainId,
            walletClientType = walletClientType,
            connectorType = connectorType,
            firstVerifiedAt = firstVerifiedAt,
            latestVerifiedAt = latestVerifiedAt,
        )
  }

  public data class PhoneAccount(val phoneNumber: String) : InternalLinkedAccount {
    override fun toLinkedAccount(): LinkedAccount =
        LinkedAccount.PhoneAccount(phoneNumber = phoneNumber)
  }

  public data class EmailAccount(val emailAddress: String) : InternalLinkedAccount {
    override fun toLinkedAccount(): LinkedAccount =
        LinkedAccount.EmailAccount(emailAddress = emailAddress)
  }

  public data class GoogleOAuthAccount(
      val subject: String,
      val email: String,
      val name: String?,
      val firstVerifiedAt: Int?,
      val latestVerifiedAt: Int?
  ) : InternalLinkedAccount {
    override fun toLinkedAccount(): LinkedAccount =
        LinkedAccount.GoogleOAuthAccount(
            subject = subject,
            email = email,
            name = name,
            firstVerifiedAt = firstVerifiedAt,
            latestVerifiedAt = latestVerifiedAt
        )
  }

  public data class TwitterOAuthAccount(
      val subject: String,
      val username: String,
      val name: String?,
      val email: String?,
      val profilePictureUrl: String?,
      val firstVerifiedAt: Int?,
      val latestVerifiedAt: Int?
  ) : InternalLinkedAccount {
    override fun toLinkedAccount(): LinkedAccount =
        LinkedAccount.TwitterOAuthAccount(
            subject = subject,
            username = username,
            name = name,
            email = email,
            profilePictureUrl = profilePictureUrl,
            firstVerifiedAt = firstVerifiedAt,
            latestVerifiedAt = latestVerifiedAt
        )
  }

  public data class DiscordOAuthAccount(
      val subject: String,
      val username: String,
      val email: String?,
      val firstVerifiedAt: Int?,
      val latestVerifiedAt: Int?
  ) : InternalLinkedAccount {
    override fun toLinkedAccount(): LinkedAccount =
        LinkedAccount.DiscordOAuthAccount(
            subject = subject,
            username = username,
            email = email,
            firstVerifiedAt = firstVerifiedAt,
            latestVerifiedAt = latestVerifiedAt
        )
  }

  public fun isEmbeddedWallet(): Boolean {
    return this is EmbeddedWalletAccount
  }

  public fun isEmailAccount(): Boolean {
    return this is EmbeddedWalletAccount
  }
}

// Retrieves the user's primary wallet with the following priority: eth -> solana
// This info is typically used to pass entropy information into an embedded wallet iFrame request
public fun List<InternalLinkedAccount>.primaryWalletForEntropyInfo():
    InternalLinkedAccount.EmbeddedWalletAccount? {
  return primaryEthereumWalletAccountOrNull() ?: primarySolanaWalletAccountOrNull()
}

// TODO: UPDATE TO INCLUDE SOLANA CHECK TOO
// Embedded Ethereum wallet with hdWalletIndex == 0 is the primary wallet
public fun List<InternalLinkedAccount>.primaryEthereumWalletAccountOrNull():
    InternalLinkedAccount.EmbeddedEthereumWalletAccount? {
  return this.filterIsInstance<InternalLinkedAccount.EmbeddedEthereumWalletAccount>().firstOrNull {
    it.hdWalletIndex == 0
  }
}

// Embedded Solana wallet with hdWalletIndex == 0 is the primary wallet
public fun List<InternalLinkedAccount>.primarySolanaWalletAccountOrNull():
    InternalLinkedAccount.EmbeddedSolanaWalletAccount? {
  return this.filterIsInstance<InternalLinkedAccount.EmbeddedSolanaWalletAccount>().firstOrNull {
    it.hdWalletIndex == 0
  }
}

public fun List<InternalLinkedAccount>.embeddedEthereumWalletAccounts():
    List<InternalLinkedAccount.EmbeddedEthereumWalletAccount> {
  return this.filterIsInstance<InternalLinkedAccount.EmbeddedEthereumWalletAccount>()
}

public fun List<InternalLinkedAccount>.embeddedSolanaWalletAccounts():
    List<InternalLinkedAccount.EmbeddedSolanaWalletAccount> {
  return this.filterIsInstance<InternalLinkedAccount.EmbeddedSolanaWalletAccount>()
}

public fun InternalLinkedAccount.EmbeddedWalletAccount.toEmbeddedWalletDetails(
    entropyWallet: InternalLinkedAccount.EmbeddedWalletAccount
): EmbeddedWalletDetails {
  return EmbeddedWalletDetails(
      id = id,
      address = address,
      chainId = chainId,
      recoveryMethod = recoveryMethod,
      // Chain type and HD index of the target wallet
      hdWalletIndex = hdWalletIndex,
      chainType = chainTypeString,
      entropyWalletDetails = entropyWallet.toEntropyWalletDetails(),
      )
}

public fun InternalLinkedAccount.EmbeddedWalletAccount.toEntropyWalletDetails(
): EmbeddedWalletDetails.EntropyWalletDetails {
    return EmbeddedWalletDetails.EntropyWalletDetails(
        address = address,
        entropyId = entropyId,
        entropyIdVerifier = entropyIdVerifier
    )
}

public val InternalLinkedAccount.EmbeddedWalletAccount.privyStack: PrivyStack
  get() = if (recoveryMethod == "privy-v2" && id != null) PrivyStack.Unified else PrivyStack.Legacy

// Generic term for "wallet address" on Ethereum or "public key" on Solana
private val InternalLinkedAccount.EmbeddedWalletAccount.entropyId: String
  get() = address

// Source of the `entropyId` property.
// ethereum-address-verifier: the entropyId is the address of the Ethereum wallet derived
// at index 0 for this entropy
// solana-address-verifier: the entropyId is a the address of the Solana wallet derived
// at index 0 for this entropy
private val InternalLinkedAccount.EmbeddedWalletAccount.entropyIdVerifier: String
  get() =
      when (this) {
        is InternalLinkedAccount.EmbeddedEthereumWalletAccount -> "ethereum-address-verifier"
        is InternalLinkedAccount.EmbeddedSolanaWalletAccount -> "solana-address-verifier"
      }

private val InternalLinkedAccount.EmbeddedWalletAccount.chainTypeString: String
  get() =
      when (this) {
        is InternalLinkedAccount.EmbeddedEthereumWalletAccount -> "ethereum"
        is InternalLinkedAccount.EmbeddedSolanaWalletAccount -> "solana"
      }

