package io.privy.auth

import kotlin.jvm.JvmInline

/**
 * Represents the current authentication state of the user within the SDK.
 * Consumers can observe instances of this sealed interface to react to changes
 * in the user's authentication status.
 */
public sealed interface AuthState {
  /**
   * When the PrivySDK is initialized, the user's authentication state will initially be [AuthState.NotReady].
   * Privy attempts to asynchronously determine the user's authenticated state. Call [awaitReady] to
   * suspend execution until Privy determines the user's auth state, at which point you can access
   * [Privy.authState], which will be guaranteed to be something other than [AuthState.NotReady]
   */
  public data object NotReady : AuthState

  /**
   * A state that represents a prior authenticated session which could not be
   * immediately verified. This is likely due to a lack of network connectivity
   * or a temporary service issue. The user is considered authenticated but their
   * session's validity is currently unconfirmed.
   *
   * It is defined as a data class to allow a backwards compatible changes in the future
   * in case we need to expose associated values.
   *
   * @param _dummy A placeholder to satisfy the data class constructor requirement.
   *
   */
  public data class AuthenticatedUnverified(private val _dummy: Unit) : AuthState

  /**
   * The user is currently unauthenticated. This state indicates that
   * there is no active session, and the user needs to log in.
   */
  public data object Unauthenticated : AuthState

  /**
   * The user is authenticated and has a valid, currently verified authentication session.
   *
   * @param user The [PrivyUser] object representing the authenticated user.
   */
  @JvmInline
  public value class Authenticated(public val user: PrivyUser) : AuthState

  /**
   * Checks if the current authentication state indicates an authenticated user.
   *
   * @return `true` if the state is [Authenticated], `false` otherwise.
   */
  public fun isAuthenticated(): Boolean {
    return this is Authenticated
  }
}