package io.privy.auth

import io.privy.auth.customAuth.LoginWithCustomAuth
import io.privy.auth.email.LoginWithEmail
import io.privy.auth.internal.InternalAuthSession
import io.privy.auth.oAuth.LoginWithOAuth
import io.privy.auth.passkey.LoginWithPasskey
import io.privy.auth.siwe.LoginWithSiwe
import io.privy.auth.siws.LoginWithSiws
import io.privy.auth.sms.LoginWithSms
import kotlinx.coroutines.flow.StateFlow

/**
 * Serves as the main entrypoint for everything Auth related. Should ideally be the only
 * entrypoint to the "authentication" module.
 *
 * If we ever want to publicly expose this, we can split this interface into "sdk internal" vs
 * "public" methods
 */
public interface PrivyAuth {
  // Once we remove Privy.isReady, we can remove this as well
  public val isReady: Boolean

  /**
   * The privy user object. Will be non null if the user is not authenticated.
   */
  public val user: PrivyUser?

  /**
   * A state flow that can be used to check current auth state or can be subscribed to for auth
   * state updates
   */
  public val authStateUpdates: StateFlow<AuthState>

  // Entry point for SMS auth
  public val sms: LoginWithSms

  // Entry point for Email auth
  public val email: LoginWithEmail

  // Entry point for third-party auth sign in
  public val customAuth: LoginWithCustomAuth

  // Entry point for sign in with ethereum
  public val siwe: LoginWithSiwe

  // Entry point for sign in with solana
  public val siws: LoginWithSiws

  // Entry point for login in with oauth
  public val oAuth: LoginWithOAuth

  // Entry point for passkey authentication
  public val passkey: LoginWithPasskey

  /**
   * A suspending function that returns the user's authenticated state after ensuring
   * all SDK initialization tasks have completed running, such as restoring / refreshing a prior
   * authenticated session.
   */
  public suspend fun getAuthState(): AuthState

  /**
   * If the PrivySDK was initialized while the device had no network connectivity, the [AuthState]
   * might remain as [AuthState.AuthenticatedUnverified]. Whenever you determine the device regains connectivity,
   * call [onNetworkRestored], which will attempt to restore the prior user session. After calling
   * this, check [Privy.authState] to check the user's authenticated state.
   *
   * A call to [onNetworkRestored] after the user's auth state had already been determined will
   * result in a no op, so you can safely call this whenever.
   */
  public suspend fun onNetworkRestored()

  /**
   * Checks if there are persisted credentials. This does not mean the user is authenticated or that
   * there is a valid user session. It simply suggests there was an authenticated user in a prior
   * session, and this session could potentially be used to re-authenticate the user.
   *
   * When there is no network connectivity, [Privy.authState] could be [AuthState.NotReady],
   * at which point this method could be used to determine if there is potentially an authenticated user.
   */
  public suspend fun hasPersistedAuthCredentials(): Boolean

  /**
   * Logs the user out. The PrivyUser will be set to null and [AuthState] will be updated to
   * [AuthState.Unauthenticated]
   */
  public suspend fun logout()

  // Once we remove Privy.awaitReady, we can remove this as well
  public suspend fun awaitInitializationComplete()

  /**
   * Checks if the current session is still valid. If the session is expired, it attempts to refresh
   * it.
   *
   * @return Result.success with a valid, refreshed session if needed, or the existing session if
   *   it's still valid. Result.failure if there is no current session to refresh or if the refresh
   *   fails.
   */
  public suspend fun refreshSessionIfNeeded(): Result<InternalAuthSession>

  // Helper function to ensure user is authenticated before making wallet call
  // If authenticated: returns result.success with associated session
  // If unauthenticated: returns result.failure
  public suspend fun <T> ensureAuthenticated(
    onAuthenticated: suspend (InternalAuthSession) -> Result<T>
  ): Result<T>
}