package io.privy.auth.passkey

import io.privy.auth.PrivyUser

/**
 * Interface for handling passkey authentication flows using FIDO2/WebAuthn.
 *
 * Passkeys provide passwordless authentication using biometric or device credentials. This
 * interface provides methods for signing up and logging in with passkeys.
 */
public interface LoginWithPasskey {
  /**
   * Signs up a new user with a passkey.
   *
   * This initiates the passkey creation flow:
   * 1. Requests registration challenge from server
   * 2. Prompts user to create passkey using biometrics/PIN
   * 3. Registers the credential with server
   * 4. Returns authenticated user
   *
   * @param relyingParty The domain that will own this credential (e.g., "yourdomain.com"). Must
   *   match your Digital Asset Links configuration.
   * @param displayName Optional user-friendly name shown in passkey selector. If not provided,
   *   defaults to a generic name.
   * @return Result containing authenticated PrivyUser on success, or PasskeyException on failure.
   */
  public suspend fun signup(
      relyingParty: String,
      displayName: String? = null,
  ): Result<PrivyUser>

  /**
   * Logs in an existing user with their passkey.
   *
   * This initiates the passkey authentication flow:
   * 1. Requests authentication challenge from server
   * 2. Shows passkey selector to user
   * 3. User authenticates using biometrics/PIN
   * 4. Verifies credential with server
   * 5. Returns authenticated user
   *
   * @param relyingParty The domain that owns the credential (e.g., "yourdomain.com"). Must match
   *   your Digital Asset Links configuration.
   * @return Result containing authenticated PrivyUser on success, or PasskeyException on failure.
   */
  public suspend fun login(relyingParty: String): Result<PrivyUser>
}
