package io.privy.auth.siwe

import io.privy.auth.PrivyUser

/** Interface for handling Sign-In with Ethereum (SIWE) authentication flows. */
public interface LoginWithSiwe {
  /**
   * Generates a SIWE message for the user to sign to prove ownership of the wallet.
   *
   * @param params Specifies the app domain, app URI, chain ID, and wallet address to build the SIWE
   *   message.
   * @return A Result containing a unique SIWE message as a string if successful, or a failure with
   *   an error.
   */
  public suspend fun generateMessage(
      params: SiweMessageParams,
  ): Result<String>

  /**
   * Logs in the user with the SIWE signature.
   *
   * @param signature The SIWE signature generated from the EIP-191 personal sign request.
   * @param metadata Optional additional metadata specifying wallet client and connector type.
   * @return A Result containing the authenticated [PrivyUser] if successful, or a failure with an
   *   error.
   */
  public suspend fun login(
      message: String,
      signature: String,
      params: SiweMessageParams,
      metadata: WalletLoginMetadata? = null,
  ): Result<PrivyUser>

  /**
   * Links the user's external Ethereum wallet to their account. The user must be authenticated
   * prior to calling this method.
   *
   * @param signature The SIWE signature generated from the EIP-191 personal sign request.
   * @param metadata Optional additional metadata specifying wallet client and connector type. If
   *   omitted, the metadata specified in [generateMessage] will be used, if applicable.
   * @return A Result indicating success (Unit) or failure with an error if linking fails or the
   *   user is not authenticated.
   */
  public suspend fun link(
      message: String,
      signature: String,
      params: SiweMessageParams,
      metadata: WalletLoginMetadata? = null,
  ): Result<Unit>

  /**
   * Unlinks an Ethereum wallet from the currently authenticated user's account.
   *
   * @param address The Ethereum wallet address to unlink
   * @return A Result indicating success (Unit) or failure with an error if unlinking fails
   */
  public suspend fun unlink(address: String): Result<Unit>

  // Deprecated methods
  @Deprecated(
      message = "Use generateMessage() instead, which has identical functionality but simpler naming",
      replaceWith = ReplaceWith("generateMessage(params)")
  )
  public suspend fun generateSiweMessage(params: SiweMessageParams): Result<String>

  @Deprecated(
      message = "Use login() instead, which has identical functionality but simpler naming",
      replaceWith = ReplaceWith("login(message, signature, params, metadata)")
  )
  public suspend fun loginWithSiwe(
      message: String,
      signature: String,
      params: SiweMessageParams,
      metadata: WalletLoginMetadata?,
  ): Result<PrivyUser>

  @Deprecated(
      message = "Use link() instead, which has identical functionality but simpler naming",
      replaceWith = ReplaceWith("link(message, signature, params, metadata)")
  )
  public suspend fun linkWithSiwe(
      message: String,
      signature: String,
      params: SiweMessageParams,
      metadata: WalletLoginMetadata?,
  ): Result<Unit>

  @Deprecated(
      message = "Use unlink() instead, which has identical functionality but simpler naming",
      replaceWith = ReplaceWith("unlink(address)")
  )
  public suspend fun unlinkWallet(address: String): Result<Unit>
}
