package io.privy.auth.siws

import io.privy.auth.PrivyUser
import io.privy.auth.siwe.WalletLoginMetadata

/** Interface for handling Sign-In with Solana (SIWS) authentication flows. */
public interface LoginWithSiws {
  /**
   * Generates a SIWS message for the user to sign to prove ownership of the Solana wallet.
   *
   * @param params Specifies the app domain, app URI, and wallet address to build the SIWS
   *   message.
   * @return A Result containing a unique SIWS message as a string if successful, or a failure with
   *   an error.
   */
  public suspend fun generateMessage(
      params: SiwsMessageParams,
  ): Result<String>

  /**
   * Logs in the user with the SIWS signature.
   *
   * @param message The SIWS message that was signed by the user's wallet.
   * @param signature The SIWS signature generated from signing the message with the Solana wallet.
   * @param params The same parameters used to generate the SIWS message.
   * @param metadata Optional additional metadata specifying wallet client and connector type.
   * @return A Result containing the authenticated [PrivyUser] if successful, or a failure with an
   *   error.
   */
  public suspend fun login(
      message: String,
      signature: String,
      params: SiwsMessageParams,
      metadata: WalletLoginMetadata? = null,
  ): Result<PrivyUser>

  /**
   * Links a Solana wallet to an existing authenticated user using SIWS.
   *
   * @param message The SIWS message that was signed by the user's wallet.
   * @param signature The SIWS signature generated from signing the message with the Solana wallet.
   * @param params The same parameters used to generate the SIWS message.
   * @param metadata Optional additional metadata specifying wallet client and connector type.
   * @return A Result containing Unit if successful, or a failure with an error.
   */
  public suspend fun link(
      message: String,
      signature: String,
      params: SiwsMessageParams,
      metadata: WalletLoginMetadata? = null,
  ): Result<Unit>

  /**
   * Unlinks the specified Solana wallet from the currently authenticated user's account.
   *
   * @param address The Solana wallet address to unlink from the user.
   * @return A Result containing Unit if successful, or a failure with an error.
   */
  public suspend fun unlink(address: String): Result<Unit>
}
