package io.privy.auth.passkey

import io.privy.auth.PrivyUser

/**
 * Interface for handling passkey authentication flows using FIDO2/WebAuthn.
 *
 * Passkeys provide passwordless authentication using biometric or device credentials. This
 * interface provides methods for signing up and logging in with passkeys.
 */
public interface LoginWithPasskey {
  /**
   * Signs up a new user with a passkey.
   *
   * This initiates the passkey creation flow:
   * 1. Requests registration challenge from server
   * 2. Prompts user to create passkey using biometrics/PIN
   * 3. Registers the credential with server
   * 4. Returns authenticated user
   *
   * @param relyingParty The domain that will own this credential (e.g., "yourdomain.com"). Must
   *   match your Digital Asset Links configuration.
   * @param displayName Optional user-friendly name shown in passkey selector. If not provided,
   *   defaults to a generic name.
   * @return Result containing authenticated PrivyUser on success, or PasskeyException on failure.
   */
  public suspend fun signup(
      relyingParty: String,
      displayName: String? = null,
  ): Result<PrivyUser>

  /**
   * Logs in an existing user with their passkey.
   *
   * This initiates the passkey authentication flow:
   * 1. Requests authentication challenge from server
   * 2. Shows passkey selector to user
   * 3. User authenticates using biometrics/PIN
   * 4. Verifies credential with server
   * 5. Returns authenticated user
   *
   * @param relyingParty The domain that owns the credential (e.g., "yourdomain.com"). Must match
   *   your Digital Asset Links configuration.
   * @return Result containing authenticated PrivyUser on success, or PasskeyException on failure.
   */
  public suspend fun login(relyingParty: String): Result<PrivyUser>

  /**
   * Links a passkey to the currently authenticated user's account.
   *
   * This initiates the passkey creation flow for an existing user:
   * 1. Verifies user is authenticated
   * 2. Requests link challenge from server (authenticated)
   * 3. Prompts user to create passkey using biometrics/PIN
   * 4. Links the credential to the user's account
   *
   * @param relyingParty The domain that will own this credential (e.g., "yourdomain.com"). Must
   *   match your Digital Asset Links configuration.
   * @param displayName Optional user-friendly name shown in passkey selector. If not provided,
   *   defaults to a generic name.
   * @return Result containing Unit on success, or error on failure.
   */
  public suspend fun link(
      relyingParty: String,
      displayName: String? = null,
  ): Result<Unit>

  /**
   * Unlinks a passkey from the currently authenticated user's account.
   *
   * This removes the specified passkey credential from the user's account:
   * 1. Verifies user is authenticated
   * 2. Requests unlink from server
   * 3. Removes the credential association
   *
   * @param credentialId The credential ID of the passkey to unlink
   * @return Result containing Unit on success, or error on failure.
   */
  public suspend fun unlink(credentialId: String): Result<Unit>
}
