package io.privy.auth

import io.privy.auth.customAuth.TokenProvider
import io.privy.auth.internal.InternalAuthSession
import io.privy.auth.otp.OtpRequestType
import kotlinx.coroutines.flow.StateFlow

public interface AuthManager {
  public val privyUser: PrivyUser

  // A state flow that can be used to subscribe to auth state updates
  //public val authState: StateFlow<AuthState>

  // public val internalAuthState: StateFlow<InternalAuthState>

  // For custom auth refresh flow
  public fun setTokenProvider(tokenProvider: TokenProvider)

  /**
   * Sends OTP to the requested destination
   *
   * @param otpRequestType Either email or SMS
   * @return Result type with no associated value representing if OTP was sent successfully or not
   */
  public suspend fun sendOtp(otpRequestType: OtpRequestType): Result<Unit>

  public suspend fun login(loginType: LoginType): Result<PrivyUser>

  // Helper function to return a value if user is authenticated,
  // or else return the provided default value
  public fun <T> authenticatedOrDefault(
    onAuthenticated: (InternalAuthSession) -> T,
    default: () -> T
    ): T

  // Helper function to ensure user is authenticated before making wallet call
  // If authenticated: returns result.success with associated session
  // If unauthenticated: returns result.failure
  public suspend fun <T> ensureAuthenticated(onAuthenticated: suspend (InternalAuthSession) -> Result<T>): Result<T>

  /**
   * Force refreshes existing session.
   *
   * @return
   * Result.success with valid, refreshed session or
   * Result.failure if there is no current session to refresh or if refresh fails
   */
  public suspend fun refreshSession(): Result<InternalAuthSession>

  /**
   * Refreshes existing session if refresh is needed.
   *
   * @return
   * Result.success with valid session
   * Result.failure if there is no current session to refresh or if refresh fails
   */
  public suspend fun refreshSessionIfNeeded(): Result<InternalAuthSession>

  /**
   * Returns the access token if the user is authenticated
   * If the current access token is expired, this will attempt to refresh
   * the session, and return the updated access token.
   *
   * If the user is not authenticated, will return null
   */
  // TODO: impl this
  // public suspend fun getAccessToken(): String?

  public suspend fun logout()
}