package io.privy.auth.internal

import io.privy.auth.LinkedAccount
import io.privy.wallet.EmbeddedWalletDetails

public data class InternalAuthSession(
  val user: InternalPrivyUser,
  val accessToken: String,
  val refreshToken: String,
  val loginMethod: LoginMethod,
  val sessionUpdateAction: SessionUpdateAction,
)

public enum class SessionUpdateAction {
  Set, Clear, Ignore;

  public companion object {
    public fun fromString(value: String): SessionUpdateAction {
      return when (value.lowercase()) {
        "set" -> Set
        "clear" -> Clear
        "ignore" -> Ignore
        else -> Set
      }
    }
  }
}

public data class InternalPrivyUser(
  val id: String,
  val linkedAccounts: List<InternalLinkedAccount>,
) {
  public fun hasEmbeddedWallet(): Boolean {
    val embeddedWallet = linkedAccounts.firstOrNull { it.isEmbeddedWallet() }
    return embeddedWallet != null
  }
}

/**
 * Internal domain entities for LinkedAccounts.
 * Will almost be 1:1 mapping with external LinkedAccounts,
 * but good to have separation between public entities and
 * storage entities
 */
public sealed interface InternalLinkedAccount {
  public fun toLinkedAccount(): LinkedAccount

  public data class CustomAuth(
    val customUserId: String,
    //TODO: UPDATE THESE TO DATE TYPE
    val firstVerifiedAt: Int?,
    val latestVerifiedAt: Int?
  ) : InternalLinkedAccount {
    override fun toLinkedAccount(): LinkedAccount = LinkedAccount.CustomAuth(
      customUserId = customUserId,
      firstVerifiedAt = firstVerifiedAt,
      latestVerifiedAt = latestVerifiedAt,
    )
  }

  public sealed interface EmbeddedWalletAccount: InternalLinkedAccount {
    public val address: String
    public val chainId: String?
    public val recoveryMethod: String?
    public val hdWalletIndex: Int
  }

  public data class EmbeddedEthereumWalletAccount(
    override val address: String,
    override val chainId: String?,
    override val recoveryMethod: String?,
    override val hdWalletIndex: Int,
    val firstVerifiedAt: Int?,
    val latestVerifiedAt: Int?,
  ): EmbeddedWalletAccount {
    override fun toLinkedAccount(): LinkedAccount = LinkedAccount.EmbeddedEthereumWalletAccount(
      address = address,
      chainId = chainId,
      recoveryMethod = recoveryMethod,
      hdWalletIndex = hdWalletIndex,
    )
  }

  public data class EmbeddedSolanaWalletAccount(
    override val address: String,
    override val chainId: String?,
    override val recoveryMethod: String?,
    override val hdWalletIndex: Int,
  ): EmbeddedWalletAccount {
    override fun toLinkedAccount(): LinkedAccount = LinkedAccount.EmbeddedSolanaWalletAccount(
      address = address,
      chainId = chainId,
      recoveryMethod = recoveryMethod,
      hdWalletIndex = hdWalletIndex,
    )
  }

  public data class PhoneAccount(
    val phoneNumber: String
  ): InternalLinkedAccount {
    override fun toLinkedAccount(): LinkedAccount = LinkedAccount.PhoneAccount(
      phoneNumber = phoneNumber
    )
  }

  public data class EmailAccount(
    val emailAddress: String
  ): InternalLinkedAccount {
    override fun toLinkedAccount(): LinkedAccount = LinkedAccount.EmailAccount(
      emailAddress = emailAddress
    )
  }

  public fun isEmbeddedWallet(): Boolean {
    return this is EmbeddedWalletAccount
  }
}

// Retrieves the user's primary wallet with the following priority: eth -> solana
// This info is typically used to pass entropy information into an embedded wallet iFrame request
public fun List<InternalLinkedAccount>.primaryWalletForEntropyInfo(): InternalLinkedAccount.EmbeddedWalletAccount? {
  return primaryEthereumWalletAccountOrNull() ?: primarySolanaWalletAccountOrNull()
}

// TODO: UPDATE TO INCLUDE SOLANA CHECK TOO
// Embedded Ethereum wallet with hdWalletIndex == 0 is the primary wallet
public fun List<InternalLinkedAccount>.primaryEthereumWalletAccountOrNull(): InternalLinkedAccount.EmbeddedEthereumWalletAccount? {
  return this
    .filterIsInstance<InternalLinkedAccount.EmbeddedEthereumWalletAccount>()
    .firstOrNull { it.hdWalletIndex == 0 }
}

// Embedded Solana wallet with hdWalletIndex == 0 is the primary wallet
public fun List<InternalLinkedAccount>.primarySolanaWalletAccountOrNull(): InternalLinkedAccount.EmbeddedSolanaWalletAccount? {
  return this
    .filterIsInstance<InternalLinkedAccount.EmbeddedSolanaWalletAccount>()
    .firstOrNull { it.hdWalletIndex == 0 }
}

public fun List<InternalLinkedAccount>.embeddedEthereumWalletAccounts():
        List<InternalLinkedAccount.EmbeddedEthereumWalletAccount> {
  return this.filterIsInstance<InternalLinkedAccount.EmbeddedEthereumWalletAccount>()
}

public fun List<InternalLinkedAccount>.embeddedSolanaWalletAccounts():
        List<InternalLinkedAccount.EmbeddedSolanaWalletAccount> {
  return this.filterIsInstance<InternalLinkedAccount.EmbeddedSolanaWalletAccount>()
}

public fun InternalLinkedAccount.EmbeddedWalletAccount.toEmbeddedWalletDetails(
  primaryEmbeddedWallet: InternalLinkedAccount.EmbeddedWalletAccount
): EmbeddedWalletDetails {
  return EmbeddedWalletDetails(
    address = address,
    chainId = chainId,
    recoveryMethod = recoveryMethod,
    // Chain type and HD index of the target wallet
    hdWalletIndex = hdWalletIndex,
    chainType = chainTypeString,
    primaryWalletDetails = EmbeddedWalletDetails.PrimaryWalletDetails(
      address = primaryEmbeddedWallet.address,
      // primary wallet's entropy info (used for iFrame events)
      entropyId = primaryEmbeddedWallet.entropyId,
      entropyIdVerifier = primaryEmbeddedWallet.entropyIdVerifier
    )
  )
}

// Generic term for "wallet address" on Ethereum or "public key" on Solana
private val InternalLinkedAccount.EmbeddedWalletAccount.entropyId: String
  get() = address

// Source of the `entropyId` property.
// ethereum-address-verifier: the entropyId is the address of the Ethereum wallet derived
// at index 0 for this entropy
// solana-address-verifier: the entropyId is a the address of the Solana wallet derived
// at index 0 for this entropy
private val InternalLinkedAccount.EmbeddedWalletAccount.entropyIdVerifier: String
  get() = when (this) {
    is InternalLinkedAccount.EmbeddedEthereumWalletAccount -> "ethereum-address-verifier"
    is InternalLinkedAccount.EmbeddedSolanaWalletAccount -> "solana-address-verifier"
  }

private val InternalLinkedAccount.EmbeddedWalletAccount.chainTypeString: String
  get() = when (this) {
    is InternalLinkedAccount.EmbeddedEthereumWalletAccount -> "ethereum"
    is InternalLinkedAccount.EmbeddedSolanaWalletAccount -> "solana"
  }