package io.privy.auth

import io.privy.wallet.ethereum.EmbeddedEthereumWallet
import io.privy.wallet.solana.EmbeddedSolanaWallet

public interface PrivyUser {
  /** The unique identifier for this user. */
  public val id: String

  /**
   * A list of linked accounts for this user. A linked account can be any of the methods a user
   * authenticated with, or a users embedded wallet.
   */
  public val linkedAccounts: List<LinkedAccount>

  /**
   * The identity token for this user, if configured in the Privy dashboard.
   *
   * This token is an optional JWT provided by Privy when identity token generation is enabled in
   * the dashboard settings. It returns null if the user is unauthenticated or if the feature is not
   * configured.
   */
  public val identityToken: String?
  /**
   * A list of the user's embedded Ethereum wallets. These wallets expose a "provider" instance
   * which can be used to take wallet actions.
   */
  public val embeddedEthereumWallets: List<EmbeddedEthereumWallet>

  /**
   * A list of the user's embedded Solana wallets. These wallets expose a "provider" instance which
   * can be used to take wallet actions.
   */
  public val embeddedSolanaWallets: List<EmbeddedSolanaWallet>

  /**
   * Creates an embedded Ethereum wallet for the user.
   *
   * @param allowAdditional Ethereum embedded wallets are
   *   [hierarchical deterministic (HD) wallets](https://www.ledger.com/academy/crypto/what-are-hierarchical-deterministic-hd-wallets),
   *   and a user's seed entropy can support multiple separate embedded wallets. If a user already
   *   has a wallet and you'd like to create additional HD wallets for them, pass in true for the
   *   allowAdditional parameter. Defaults to false.
   * @return A [Result] object containing the newly created [EmbeddedEthereumWallet] if successful,
   *   or Result.failure if:
   *
   * The user is no longer authenticated If a user already has 9 or more wallets If the network call
   * to create the wallet fails If a user already has an embedded Ethereum wallet and
   * allowAdditional is not set to true.
   */
  public suspend fun createEthereumWallet(
      allowAdditional: Boolean = false
  ): Result<EmbeddedEthereumWallet>

  /**
   * Creates an embedded Solana wallet for the user.
   *
   * @return A [Result] object containing the newly created [EmbeddedSolanaWallet] if successful, or
   *   Result.failure if:
   *
   * The user is no longer authenticated If the network call to create the wallet fails If a user
   * already has an embedded Solana wallet
   */
  public suspend fun createSolanaWallet(
    allowAdditional: Boolean = false
  ): Result<EmbeddedSolanaWallet>

  /**
   * Fetches the latest user info. If the user's session is expired, the session will first be
   * refreshed.
   *
   * @return A [Result] object with no associated type, indicating if refreshing the user was
   *   successful or not
   */
  public suspend fun refresh(): Result<Unit>

  /**
   * Retrieves the latest access token for the authenticated user.
   *
   * This function ensures the session is valid before returning the access token. If the session is
   * expired, it attempts to refresh it before extracting the token.
   *
   * @return A [Result] containing the access token if successful, or a failure if the session
   *   cannot be refreshed.
   */
  public suspend fun getAccessToken(): Result<String>
}
