package io.privy.wallet

import io.privy.auth.AuthStateRepository
import io.privy.auth.EmbeddedWalletException
import io.privy.auth.internal.InternalAuthSession
import io.privy.auth.internal.InternalAuthState
import io.privy.auth.internal.InternalLinkedAccount
import io.privy.auth.internal.primaryWalletForEntropyInfo
import io.privy.auth.internal.toEmbeddedWalletDetails
import io.privy.logging.PrivyLogger
import io.privy.network.NetworkStateManager
import io.privy.network.NoNetworkException
import io.privy.network.isConfirmedDisconnected
import io.privy.wallet.ethereum.EthereumRpcRequest
import io.privy.wallet.ethereum.EthereumRpcResponse
import io.privy.wallet.solana.SolanaRpcRequest
import io.privy.wallet.solana.SolanaSignMessageResponse
import io.privy.wallet.webview.IFrameResponse
import io.privy.wallet.webview.WebViewHandler
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.launch
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import me.tatarka.inject.annotations.Inject

// The layer between KMP SDK and WebViewHandler
// In charge of managing web view states so that it is abstracted to callers
@Inject
public class RealEmbeddedWalletManager(
    private val webViewHandler: WebViewHandler,
    private val privyLogger: PrivyLogger,
    private val authStateRepository: AuthStateRepository,
    private val networkStateManager: NetworkStateManager,
) : EmbeddedWalletManager {
  private val scope = CoroutineScope(SupervisorJob() + Dispatchers.Default)

  private var embeddedWalletState =
      MutableStateFlow<EmbeddedWalletState>(EmbeddedWalletState.Disconnected)

  private val connectWalletMutex = Mutex()

  init {
    subscribeToAuthStateUpdates()
  }

  override suspend fun createEthereumWallet(
      accessToken: String,
      existingSolanaWalletAddress: String?,
  ): Result<String> {
    return when (val createWalletResponse =
        webViewHandler.createEthereumWallet(
            accessToken = accessToken, existingSolanaWalletAddress = existingSolanaWalletAddress)) {
      is IFrameResponse.Error -> {
        // Some other error - surface it as failure
        val errorMessage = createWalletResponse.message
        privyLogger.error("Error creating Ethereum wallet: $errorMessage")
        Result.failure(EmbeddedWalletException(errorMessage))
      }
      is IFrameResponse.Success -> {
        val newWalletAddress = createWalletResponse.value.data.address
        privyLogger.debug("Newly created Ethereum wallet address: $newWalletAddress")

        Result.success(newWalletAddress)
      }
    }
  }

  override suspend fun createAdditionalWallet(
      accessToken: String,
      primaryWalletAddress: String,
      hdWalletIndex: Int,
  ): Result<String> {
    val response =
        webViewHandler.createAdditionalWallet(
            accessToken = accessToken,
            primaryWalletAddress = primaryWalletAddress,
            hdWalletIndex = hdWalletIndex,
        )

    return when (response) {
      is IFrameResponse.Error -> {
        // Some other error - surface it as failure
        val errorMessage = response.message
        privyLogger.error("Error creating additional wallet: $errorMessage")
        Result.failure(EmbeddedWalletException(errorMessage))
      }
      is IFrameResponse.Success -> {
        val newWalletAddress = response.value.data.address
        privyLogger.debug("Newly created wallet address: $newWalletAddress")
        Result.success(newWalletAddress)
      }
    }
  }

  override suspend fun createSolanaWallet(
      accessToken: String,
      existingEthereumWalletAddress: String?,
  ): Result<String> {
    return when (val createWalletResponse =
        webViewHandler.createSolanaWallet(
            accessToken = accessToken,
            existingEthereumWalletAddress = existingEthereumWalletAddress)) {
      is IFrameResponse.Error -> {
        // Some other error - surface it as failure
        val errorMessage = createWalletResponse.message
        privyLogger.error("Error creating Solana wallet: $errorMessage")
        Result.failure(EmbeddedWalletException(errorMessage))
      }
      is IFrameResponse.Success -> {
        val newWalletAddress = createWalletResponse.value.data.publicKey
        privyLogger.debug("Newly created Solana wallet address: $newWalletAddress")

        Result.success(newWalletAddress)
      }
    }
  }

  override suspend fun ethereumRpc(
      embeddedWalletDetails: EmbeddedWalletDetails,
      request: EthereumRpcRequest
  ): Result<EthereumRpcResponse> {
    return awaitConnected()
        .fold(
            onSuccess = { authenticatedAndConnectedState ->
              val rpcResponse =
                  webViewHandler.ethereumRpc(
                      accessToken = authenticatedAndConnectedState.accessToken,
                      embeddedWalletDetails = embeddedWalletDetails,
                      request = request,
                  )

              return when (rpcResponse) {
                is IFrameResponse.Success -> {
                  privyLogger.info("Wallet RPC response: ${rpcResponse.value.data}")
                  Result.success(
                      EthereumRpcResponse(
                          method = rpcResponse.value.data.response.method,
                          data = rpcResponse.value.data.response.data))
                }
                is IFrameResponse.Error -> {
                  // Some other error - surface it as failure
                  val errorMessage = rpcResponse.message
                  privyLogger.error("Error sending rpc request: $errorMessage")
                  Result.failure(EmbeddedWalletException(errorMessage))
                }
              }
            },
            onFailure = { Result.failure(it) })
  }

  override suspend fun solanaRpc(
      embeddedWalletDetails: EmbeddedWalletDetails,
      request: SolanaRpcRequest
  ): Result<SolanaSignMessageResponse> {
    return awaitConnected()
        .fold(
            onSuccess = { authenticatedAndConnectedState ->
              // For now - specifying generic as SolanaSignMessageResponseData is fine because
              // we only support sign message, but in the future, we'll have to actually have this
              // be generic when we support other rpc methods
              return webViewHandler
                  .solanaRpc(
                      accessToken = authenticatedAndConnectedState.accessToken,
                      embeddedWalletDetails = embeddedWalletDetails,
                      request = request,
                  )
                  .onSuccess { privyLogger.info("Solana RPC response: ${it.signature}") }
                  .onFailure {
                    privyLogger.error("Error sending Solana rpc request: ${it.message}")
                  }
            },
            onFailure = { Result.failure(it) })
  }

  private fun subscribeToAuthStateUpdates() {
    scope.launch {
      authStateRepository.internalAuthState.collectLatest { internalAuthState ->
        privyLogger.info("Auth state update received in embedded wallet manager")

        if (internalAuthState is InternalAuthState.Authenticated) {
          // if user authenticates, and they have embedded wallets, we should connect the wallet if
          // needed
          internalAuthState.session.attemptConnectWallet()
        } else {
          privyLogger.info("User is unauthenticated. Disconnecting wallet.")
          // If user is anything other than authenticated, disconnect wallet
          embeddedWalletState.value = EmbeddedWalletState.Disconnected
        }
      }
    }
  }

  override suspend fun attemptConnectWalletInBackground() {
    scope.launch {
      val internalAuthState = authStateRepository.internalAuthState.value

      if (internalAuthState is InternalAuthState.Authenticated) {
        // if user authenticates, and they have embedded wallets, we should connect the wallet if
        // needed
        internalAuthState.session.attemptConnectWallet()
      }
    }
  }

  private suspend fun InternalAuthSession.attemptConnectWallet(): Result<String> {
    val primaryEmbeddedWallet =
        this.user.linkedAccounts.primaryWalletForEntropyInfo()
            ?: return Result.failure(
                EmbeddedWalletException("User doesn't have an embedded wallet."))

    val embeddedWalletState = embeddedWalletState.value
    return if (embeddedWalletState is EmbeddedWalletState.Connected) {
      // User is already connected, return success
      Result.success(embeddedWalletState.walletAddress)
    } else {
      privyLogger.error("Embedded wallet exists and is not connected, attempting to connect.")
      connectWallet(accessToken = this.accessToken, primaryEmbeddedWallet = primaryEmbeddedWallet)
    }
  }

  // Helper function that first checks a user is authenticated
  // Then checks if the wallet is connected
  // If it isn't, it attempts to connect the wallet
  private suspend fun awaitConnected(): Result<AuthenticatedAndConnectedState> {
    ensureAuthenticated()
        .fold(
            onSuccess = { internalAuthSession ->
              return internalAuthSession.attemptConnectWallet().map { walletAddress ->
                AuthenticatedAndConnectedState(
                    accessToken = internalAuthSession.accessToken, walletAddress = walletAddress)
              }
            },
            onFailure = {
              return Result.failure(it)
            })
  }

  private data class AuthenticatedAndConnectedState(
      val accessToken: String,
      val walletAddress: String
  )

  // Helper function to ensure user is authenticated before making wallet call
  // If authenticated: returns success with associated accessToken
  // If unauthenticated: returns result failure
  private fun ensureAuthenticated(): Result<InternalAuthSession> {
    val authState = authStateRepository.internalAuthState.value

    return if (authState is InternalAuthState.Authenticated) {
      Result.success(authState.session)
    } else {
      Result.failure(exception = EmbeddedWalletException("User is not authenticated."))
    }
  }

  // TODO: One day we can move this into it's own use case and only expose this.
  private suspend fun connectWallet(
      accessToken: String,
      primaryEmbeddedWallet: InternalLinkedAccount.EmbeddedWalletAccount,
  ): Result<String> {
    val walletState = embeddedWalletState.value
    if (walletState is EmbeddedWalletState.Connected) {
      // If wallet is already connected, no op
      privyLogger.info("Wallet already connected, no op.")
      return Result.success(walletState.walletAddress)
    }

    // If there's no network, don't even attempt to connect
    if (networkStateManager.current.isConfirmedDisconnected()) {
      return Result.failure(NoNetworkException)
    }

    // Wrap this in a mutex so that we're only attempting to connect one at a time
    return connectWalletMutex.withLock {
      connectWallet(
          accessToken = accessToken,
          primaryEmbeddedWallet = primaryEmbeddedWallet,
          attemptRecovery = true,
      )
    }
  }

  // Most callers should not call this directly, and should call
  // connectWallet(accessToken, primaryWalletAddress) directly, since that function internally
  // handle
  // mutual exclusion
  private suspend fun connectWallet(
      accessToken: String,
      primaryEmbeddedWallet: InternalLinkedAccount.EmbeddedWalletAccount,
      attemptRecovery: Boolean
  ): Result<String> {
    // Check again, in case a prior caller resulted in connected state
    val walletState = embeddedWalletState.value
    if (walletState is EmbeddedWalletState.Connected) {
      // If wallet is already connected, no op
      privyLogger.info("Wallet already connected, no op.")
      return Result.success(walletState.walletAddress)
    }

    val connectWalletResponse =
        webViewHandler.connectWallet(
            accessToken = accessToken,
            embeddedWalletDetails =
                primaryEmbeddedWallet.toEmbeddedWalletDetails(
                    primaryEmbeddedWallet = primaryEmbeddedWallet),
        )

    return when (connectWalletResponse) {
      is IFrameResponse.Error -> {
        if (connectWalletResponse.type == "wallet_not_on_device" && attemptRecovery) {
          privyLogger.info("Wallet needs recovery. Attempting to recover.")
          // Wallet needs recovery
          recoverWalletThenTryConnecting(
              accessToken = accessToken, primaryEmbeddedWallet = primaryEmbeddedWallet)
        } else {
          // Some other error - surface it as failure
          val errorMessage = connectWalletResponse.message
          privyLogger.error("Error connecting wallet: $errorMessage")
          Result.failure(EmbeddedWalletException(errorMessage))
        }
      }
      is IFrameResponse.Success -> {
        privyLogger.info("Wallet '${primaryEmbeddedWallet.address}' connected!")
        // connect success! update state to connected
        val connectedWalletAddress = connectWalletResponse.value.data.entropyId
        embeddedWalletState.value = EmbeddedWalletState.Connected(connectedWalletAddress)

        Result.success(connectedWalletAddress)
      }
    }
  }

  private suspend fun recoverWalletThenTryConnecting(
      accessToken: String,
      primaryEmbeddedWallet: InternalLinkedAccount.EmbeddedWalletAccount,
  ): Result<String> {
    // try recovering
    val recoverWalletResponse =
        webViewHandler.recoverWallet(
            accessToken = accessToken,
            embeddedWalletDetails =
                primaryEmbeddedWallet.toEmbeddedWalletDetails(
                    primaryEmbeddedWallet = primaryEmbeddedWallet))

    // if success, try connecting with attemptRecovery == false
    return when (recoverWalletResponse) {
      is IFrameResponse.Error -> {
        // Recovery error, so just return the error up.
        val errorMessage = recoverWalletResponse.message
        privyLogger.error("Error recovering wallet: $errorMessage")
        Result.failure(EmbeddedWalletException(errorMessage))
      }
      is IFrameResponse.Success -> {
        privyLogger.info("Successfully recovered wallet. Attempting to connect wallet now.")
        connectWallet(
            accessToken = accessToken,
            primaryEmbeddedWallet = primaryEmbeddedWallet,
            attemptRecovery = false // important to set this to false to prevent infinite recursion
            )
      }
    }
  }
}
