package io.privy.wallet

import io.privy.appconfig.AppConfigRepository
import io.privy.auth.AuthManager
import io.privy.auth.AuthStateRepository
import io.privy.auth.EmbeddedWalletException
import io.privy.auth.flatMap
import io.privy.auth.internal.InternalAuthState
import io.privy.auth.internal.primaryWalletForEntropyInfo
import io.privy.auth.internal.privyStack
import io.privy.logging.PrivyLogger
import io.privy.wallet.creation.WalletCreatorFactory
import io.privy.wallet.ethereum.EthereumRpcRequest
import io.privy.wallet.ethereum.EthereumRpcResponse
import io.privy.wallet.rpc.WalletRpcExecutorFactory
import io.privy.wallet.solana.SolanaRpcRequest
import io.privy.wallet.solana.SolanaSignMessageResponse
import io.privy.wallet.webview.WebViewWalletConnector
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.IO
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.launch
import me.tatarka.inject.annotations.Inject

// The layer between KMP SDK and WebViewHandler
// In charge of managing web view states so that it is abstracted to callers
@Inject
public class RealEmbeddedWalletManager(
  private val privyLogger: PrivyLogger,
  private val authStateRepository: AuthStateRepository,
  private val webViewWalletConnector: WebViewWalletConnector,
  private val rpcExecutorFactory: WalletRpcExecutorFactory,
  // Inject lazily to prevent circular dependency
  private val lazyAuthManager: Lazy<AuthManager>,
  private val appConfigRepository: AppConfigRepository,
  private val walletCreatorFactory: WalletCreatorFactory,
) : EmbeddedWalletManager {
  private val authManager by lazyAuthManager
  private val scope = CoroutineScope(SupervisorJob() + Dispatchers.IO)

  init {
    scope.launch {
      authStateRepository.internalAuthState.collectLatest { internalAuthState ->
        if (internalAuthState is InternalAuthState.Authenticated) {
          // Proactively connect wallet if needed when user is authenticated
          connectWalletIfNeeded()
        }
      }
    }
  }

  override suspend fun createEthereumWallet(
    primaryEthereumWalletAddress: String?,
    ethereumWalletsCount: Int,
    primarySolanaWalletAddress: String?,
    allowAdditional: Boolean
  ): Result<String> {
    return authManager.ensureAuthenticated { authSession ->
      val accessToken = authSession.accessToken

      if (primaryEthereumWalletAddress == null) {
        // User does not have an embedded wallet yet, need to create primary wallet
        createPrimaryEthereumWallet(
          accessToken = accessToken,
          primarySolanaWalletAddress = primarySolanaWalletAddress,
        )
      } else {
        // User already has a primary wallet, create additional if allowed
        createAdditionalEthereumWalletIfAllowed(
          allowAdditional = allowAdditional,
          accessToken = accessToken,
          primaryWalletAddress = primaryEthereumWalletAddress,
          currentWalletCount = ethereumWalletsCount,
        )
      }
    }
  }

  override suspend fun createSolanaWallet(
    primarySolanaWalletAddress: String?,
    primaryEthereumWalletAddress: String?,
  ): Result<String> {
    return authManager.ensureAuthenticated { authSession ->
      val accessToken = authSession.accessToken

      if (primarySolanaWalletAddress == null) {
        // User does not have a Solana embedded wallet yet, need to create primary wallet
        createPrimarySolanaWallet(
          accessToken = accessToken,
          primaryEmbeddedEthereumWalletAddress = primaryEthereumWalletAddress,
        )
      } else {
        // User already has a primary wallet
        privyLogger.error("User already has a solana wallet.")

        Result.failure(
          exception = EmbeddedWalletException(message = "User already has a solana wallet.")
        )
      }
    }
  }

  /**
   * Internal method in charge of creating the user's / first Ethereum primary wallet (hdIndex == 0)
   * Caller of this method should verify user is authenticated and pass in a valid access token
   */
  private suspend fun createPrimaryEthereumWallet(
    accessToken: String,
    primarySolanaWalletAddress: String?,
  ): Result<String> {
    return appConfigRepository.loadAppConfig()
      .flatMap { appConfig ->
        walletCreatorFactory.create(appConfig.embeddedWalletConfig)
          .createEthereumWallet(
            accessToken = accessToken,
            existingSolanaWalletAddress = primarySolanaWalletAddress,
          )
      }
  }

  /**
   * Internal method in charge of creating the user's / first Solana primary wallet (hdIndex == 0)
   * Caller of this method should verify user is authenticated and pass in a valid access token
   */
  private suspend fun createPrimarySolanaWallet(
    accessToken: String,
    primaryEmbeddedEthereumWalletAddress: String?,
  ): Result<String> {
    return appConfigRepository.loadAppConfig()
      .flatMap { appConfig ->
        walletCreatorFactory.create(appConfig.embeddedWalletConfig)
          .createSolanaWallet(
            accessToken = accessToken,
            existingEthereumWalletAddress = primaryEmbeddedEthereumWalletAddress,
          )
      }
  }

  /**
   * Internal method in charge of creating an additional wallet for the user's if dev specifies
   * additional wallets are allowed. Caller of this method should verify user is authenticated and
   * pass in a valid access token, and should verify a user already has a primary embedded wallet
   * account.
   */
  private suspend fun createAdditionalEthereumWalletIfAllowed(
    allowAdditional: Boolean,
    accessToken: String,
    primaryWalletAddress: String,
    currentWalletCount: Int,
  ): Result<String> {
    return if (allowAdditional) {
      createAdditionalWallet(
        accessToken = accessToken,
        primaryEthereumWalletAddress = primaryWalletAddress,
        currentWalletCount = currentWalletCount,
      )
    } else {
      // Dev didn't specify allowAdditional == true, so throw an error
      val errorMessage =
        "User already has an embedded wallet. To create an additional wallet, set allowAdditional to true."
      privyLogger.error(errorMessage)

      Result.failure(exception = EmbeddedWalletException(message = errorMessage))
    }
  }

  /**
   * Internal method in charge of creating an additional wallet for the user's (hdIndex > 0) Caller
   * of this method should verify user is authenticated and pass in a valid access token, and should
   * verify a user already has a primary embedded wallet account.
   */
  private suspend fun createAdditionalWallet(
    accessToken: String,
    primaryEthereumWalletAddress: String,
    currentWalletCount: Int
  ): Result<String> {
    return appConfigRepository.loadAppConfig()
      .flatMap { appConfig ->
        walletCreatorFactory.create(appConfig.embeddedWalletConfig)
          .createAdditionalEthereumWallet(
            accessToken = accessToken,
            primaryWalletAddress = primaryEthereumWalletAddress,
            // The new wallets HD index should be the current highest wallet's index + 1
            // which is the same as the total count of embedded wallets
            hdWalletIndex = currentWalletCount,
          )
      }
  }

  override suspend fun ethereumRpc(
      embeddedWalletDetails: EmbeddedWalletDetails,
      request: EthereumRpcRequest
  ): Result<EthereumRpcResponse> {
    val rpcExecutor = rpcExecutorFactory.create(embeddedWalletDetails)

    // Get authentication token directly from AuthManager for unified stack wallets
    return authManager.refreshSessionIfNeeded()
      .map { it.accessToken }
      .flatMap { accessToken ->
        // Execute the request through the RPC executor with the auth token
        rpcExecutor.performEthRpc(
          request = request,
          embeddedWalletDetails = embeddedWalletDetails,
          accessToken = accessToken
        )
      }
  }

  override suspend fun solanaRpc(
      embeddedWalletDetails: EmbeddedWalletDetails,
      request: SolanaRpcRequest
  ): Result<SolanaSignMessageResponse> {
      val rpcExecutor = rpcExecutorFactory.create(embeddedWalletDetails)

      // Get authentication token directly from AuthManager for unified stack wallets
      // The rpcExecutor.performSolanaRpc now expects SolanaRpcRequest directly.
      return authManager.refreshSessionIfNeeded()
        .map { it.accessToken }
        .flatMap { accessToken ->
          // Execute the request through the RPC executor with the auth token
          rpcExecutor.performSolanaRpc(
            request = request, // Pass SolanaRpcRequest directly
            embeddedWalletDetails = embeddedWalletDetails,
            accessToken = accessToken
          )
        }
  }

  override fun connectWalletIfNeeded() {
    val authState = authStateRepository.internalAuthState.value

    if (authState is InternalAuthState.Authenticated) {
      authState.session.user.linkedAccounts.primaryWalletForEntropyInfo()?.let { wallet ->
        // Only legacy wallets need to be connected via webview
        if (wallet.privyStack == PrivyStack.Legacy) {
          privyLogger.debug("Proactively attempting to connect wallet if needed.")
          webViewWalletConnector.attemptConnectWalletInBackground()
        }
      }
    }
  }
}
