package io.privy.wallet.rpc

import io.privy.auth.EmbeddedWalletException
import io.privy.wallet.EmbeddedWalletDetails
import io.privy.wallet.ethereum.EthereumRpcRequest
import io.privy.wallet.ethereum.EthereumRpcResponse
import io.privy.wallet.solana.SolanaRpcRequest
import io.privy.wallet.solana.SolanaSignMessageResponse
import io.privy.wallet.walletApi.WalletApiRepository
import io.privy.wallet.walletApi.rpc.WalletApiRpcRequest
import io.privy.wallet.walletApi.rpc.WalletApiRpcResponse
import io.privy.wallet.walletApi.rpc.ethereum.WalletApiEthereumPersonalSignRpcParams
import io.privy.wallet.walletApi.rpc.ethereum.WalletApiEthereumSecp256k1SignRpcParams
import io.privy.wallet.walletApi.rpc.ethereum.WalletApiEthereumSignTypedDataV4RpcParams
import io.privy.wallet.walletApi.rpc.ethereum.EthereumTypedDataV4
import io.privy.wallet.walletApi.rpc.ethereum.UnsignedEthereumTransaction
import io.privy.wallet.walletApi.rpc.ethereum.WalletApiCAIP2
import io.privy.wallet.walletApi.rpc.ethereum.WalletApiEthereumSendTransactionRpcParams
import io.privy.wallet.walletApi.rpc.ethereum.WalletApiEthereumSignTransactionRpcParams
import io.privy.wallet.walletApi.rpc.ethereum.toWalletApi
import io.privy.wallet.walletApi.rpc.solana.WalletApiSolanaSignMessageRpcParams
import kotlinx.serialization.json.Json
import me.tatarka.inject.annotations.Inject

/**
 * An RPC execution strategy for server wallets, via the Wallets API
 */
@Inject
public class RealWalletApiRpcExecutor(
    private val walletApi: WalletApiRepository
) : WalletRpcExecutor {
    public companion object {
        public const val diQualifierName: String = "RealWalletApiRPCExecutor"
    }

    private val jsonParser = Json { ignoreUnknownKeys = true }

    /**
     * Process Ethereum RPC requests
     *
     * @param request The Ethereum RPC request to process
     * @param embeddedWalletDetails Details about the wallet making the request
     * @param accessToken Authentication token for API access
     * @return Result containing EthereumRpcResponse or failure
     */
    override suspend fun performEthRpc(
        request: EthereumRpcRequest,
        embeddedWalletDetails: EmbeddedWalletDetails,
        accessToken: String
    ): Result<EthereumRpcResponse> {
        when (request.method) {
            "personal_sign" -> {
                val message = request.params.firstOrNull()
                    ?: return Result.failure(EmbeddedWalletException("personal_sign requires a message parameter"))

                val params = WalletApiEthereumPersonalSignRpcParams(
                    message = if (isHexEncoded(message)) message.removePrefix("0x") else message,
                    encoding = if (isHexEncoded(message)) WalletApiEthereumPersonalSignRpcParams.MessageEncoding.HEX
                    else WalletApiEthereumPersonalSignRpcParams.MessageEncoding.UTF8
                )

                return embeddedWalletDetails.id?.let { walletId ->
                    walletApi.rpc(
                        request = WalletApiRpcRequest.EthereumPersonalSign(ethereumParams = params),
                        walletId = walletId,
                        token = accessToken
                    ).fold(
                        onSuccess = { rpcResponse ->
                            if (rpcResponse is WalletApiRpcResponse.EthereumPersonalSign) {
                                Result.success(
                                    EthereumRpcResponse(
                                        method = request.method,
                                        data = rpcResponse.data.signature
                                    )
                                )
                            } else {
                                Result.failure(EmbeddedWalletException("Unexpected response type for personal_sign"))
                            }
                        },
                        onFailure = { Result.failure(it) }
                    )
                } ?: Result.failure(EmbeddedWalletException("Wallet ID is required"))
            }
            "secp256k1_sign" -> {
                val hash = request.params.firstOrNull()
                    ?: return Result.failure(EmbeddedWalletException("secp256k1_sign requires a hash parameter"))

                if (!isHexEncoded(hash)) {
                    return Result.failure(EmbeddedWalletException("secp256k1_sign requires a hex-encoded hash parameter, got: $hash"))
                }

                val params = WalletApiEthereumSecp256k1SignRpcParams(hash = hash)

                return embeddedWalletDetails.id?.let { walletId ->
                    walletApi.rpc(
                        request = WalletApiRpcRequest.EthereumSecp256k1Sign(ethereumParams = params),
                        walletId = walletId,
                        token = accessToken
                    ).fold(
                        onSuccess = { rpcResponse ->
                            if (rpcResponse is WalletApiRpcResponse.EthereumSecp256k1Sign) {
                                Result.success(
                                    EthereumRpcResponse(
                                        method = request.method,
                                        data = rpcResponse.data.signature
                                    )
                                )
                            } else {
                                Result.failure(EmbeddedWalletException("Unexpected response type for secp256k1_sign"))
                            }
                        },
                        onFailure = { Result.failure(it) }
                    )
                } ?: Result.failure(EmbeddedWalletException("Wallet ID is required"))
            }
            "eth_signTypedData_v4" -> {
                if (request.params.size != 2) {
                    return Result.failure(EmbeddedWalletException("eth_signTypedData_v4 requires only address and typedData JSON string parameters"))
                }
                val typedDataJsonString = request.params[1]

                val typedData = try {
                    // Deserialize directly into EthereumTypedDataV4
                    jsonParser.decodeFromString<EthereumTypedDataV4>(typedDataJsonString)
                } catch (e: Exception) {
                    return Result.failure(EmbeddedWalletException("Failed to parse typedData JSON for eth_signTypedData_v4: ${e.message}"))
                }

                val params = WalletApiEthereumSignTypedDataV4RpcParams(typedData = typedData)

                return embeddedWalletDetails.id?.let { walletId ->
                    walletApi.rpc(
                        request = WalletApiRpcRequest.EthereumSignTypedDataV4(ethereumParams = params),
                        walletId = walletId,
                        token = accessToken
                    ).fold(
                        onSuccess = { rpcResponse ->
                            if (rpcResponse is WalletApiRpcResponse.EthereumSignTypedDataV4) {
                                Result.success(
                                    EthereumRpcResponse(
                                        method = request.method,
                                        data = rpcResponse.data.signature
                                    )
                                )
                            } else {
                                Result.failure(EmbeddedWalletException("Unexpected response type for eth_signTypedData_v4"))
                            }
                        },
                        onFailure = { Result.failure(it) }
                    )
                } ?: Result.failure(EmbeddedWalletException("Wallet ID is required for eth_signTypedData_v4"))
            }
            "eth_signTransaction" -> {
              val txJsonString = request.params.firstOrNull()
                  ?: return Result.failure(EmbeddedWalletException("eth_signTransaction requires the transaction JSON string as its first parameter"))

              // Parse the transaction JSON
              val transaction = jsonParser.decodeFromString<UnsignedEthereumTransaction>(txJsonString)
              val walletApiTransaction = transaction.toWalletApi()

              val apiTransactionParams = WalletApiRpcRequest.EthereumSignTransaction(
                  ethereumParams = WalletApiEthereumSignTransactionRpcParams(
                      transaction = walletApiTransaction
                  )
              )

              return embeddedWalletDetails.id?.let { walletId ->
                  walletApi.rpc(
                      request = apiTransactionParams,
                      walletId = walletId,
                      token = accessToken
                  ).fold(
                      onSuccess = { rpcResponse ->
                          if (rpcResponse is WalletApiRpcResponse.EthereumSignTransaction) {
                              Result.success(
                                  EthereumRpcResponse(
                                      method = request.method,
                                      data = rpcResponse.data.signedTransaction
                                  )
                              )
                          } else {
                              Result.failure(EmbeddedWalletException("Unexpected response type for eth_signTransaction"))
                          }
                      },
                      onFailure = { Result.failure(it) }
                  )
              } ?: Result.failure(EmbeddedWalletException("Wallet ID is required for eth_signTransaction"))
          }
            "eth_sendTransaction" -> {
              val txJsonString = request.params.firstOrNull()
                  ?: return Result.failure(EmbeddedWalletException("eth_sendTransaction requires the transaction JSON string as its first parameter"))

              // Parse the transaction JSON
              val transaction = jsonParser.decodeFromString<UnsignedEthereumTransaction>(txJsonString)
              val walletApiTransaction = transaction.toWalletApi()
              
              // Get chain ID from transaction or fall back to wallet details
              val chainId = transaction.chainId?.number ?: embeddedWalletDetails.chainId?.toIntOrNull()
              ?: return Result.failure(EmbeddedWalletException("Chain ID is required for eth_sendTransaction"))
              
              // Create CAIP2 chain ID for the EIP-155 chain
              val caip2 = WalletApiCAIP2.forEip155Chain(chainId)
              
              val apiTransactionParams = WalletApiRpcRequest.EthereumSendTransaction(
                  ethereumParams = WalletApiEthereumSendTransactionRpcParams(
                      transaction = walletApiTransaction
                  ),
                  caip2 = caip2
              )

              return embeddedWalletDetails.id?.let { walletId ->
                  walletApi.rpc(
                      request = apiTransactionParams,
                      walletId = walletId,
                      token = accessToken
                  ).fold(
                      onSuccess = { rpcResponse ->
                          if (rpcResponse is WalletApiRpcResponse.EthereumSendTransaction) {
                              Result.success(
                                  EthereumRpcResponse(
                                      method = request.method,
                                      data = rpcResponse.data.hash // Use transaction hash as response data
                                  )
                              )
                          } else {
                              Result.failure(EmbeddedWalletException("Unexpected response type for eth_sendTransaction"))
                          }
                      },
                      onFailure = { Result.failure(it) }
                  )
              } ?: Result.failure(EmbeddedWalletException("Wallet ID is required for eth_sendTransaction"))
          }
            "eth_sign" -> {
                if (request.params.size != 2) {
                    return Result.failure(EmbeddedWalletException("eth_sign requires 2 parameters: [address, message]"))
                }
                val message = request.params[1]

                if (!isHexEncoded(message)) {
                    return Result.failure(EmbeddedWalletException("eth_sign requires a hex-encoded message parameter, got: $message"))
                }
                // Ensure the hex message is 32 bytes (64 hex chars + 0x prefix)
                if (message.length != 66) {
                    return Result.failure(EmbeddedWalletException("eth_sign message parameter must be a 32-byte hex string (66 chars with 0x prefix), got length ${message.length}"))
                }

                val params = WalletApiEthereumSecp256k1SignRpcParams(hash = message)

                return embeddedWalletDetails.id?.let { walletId ->
                    walletApi.rpc(
                        request = WalletApiRpcRequest.EthereumSecp256k1Sign(ethereumParams = params),
                        walletId = walletId,
                        token = accessToken
                    ).fold(
                        onSuccess = { rpcResponse ->
                            if (rpcResponse is WalletApiRpcResponse.EthereumSecp256k1Sign) {
                                Result.success(
                                    EthereumRpcResponse(
                                        method = request.method,
                                        data = rpcResponse.data.signature
                                    )
                                )
                            } else {
                                Result.failure(EmbeddedWalletException("Unexpected response type for eth_sign"))
                            }
                        },
                        onFailure = { Result.failure(it) }
                    )
                } ?: Result.failure(EmbeddedWalletException("Wallet ID is required for eth_sign"))
            }

            else -> return Result.failure(EmbeddedWalletException("Unsupported Ethereum RPC method: ${request.method}"))
        }
    }

    /**
     * Process Solana RPC requests
     *
     * @param params The Solana sign message parameters
     * @param embeddedWalletDetails Details about the wallet making the request
     * @param accessToken Authentication token for API access
     * @return Result containing SolanaSignMessageResponse or failure
     */
    override suspend fun performSolanaRpc(
        params: SolanaRpcRequest,
        embeddedWalletDetails: EmbeddedWalletDetails,
        accessToken: String
    ): Result<SolanaSignMessageResponse> {
        if (params is SolanaRpcRequest.SignMessageParams) {
            // Validate message is not blank
            if (params.message.isBlank()) {
                return Result.failure(EmbeddedWalletException("Solana signMessage requires a non-blank message"))
            }

            // Convert SolanaSignMessageParams to WalletApiSolanaSignMessageRpcParams
            val apiParams = WalletApiSolanaSignMessageRpcParams(
                message = params.message, // Now correctly accessed after type check
                encoding = WalletApiSolanaSignMessageRpcParams.MessageEncoding.BASE64
            )

            return embeddedWalletDetails.id?.let { walletId ->
                walletApi.rpc(
                    request = WalletApiRpcRequest.SolanaSignMessage(solanaParams = apiParams),
                    walletId = walletId,
                    token = accessToken
                ).fold(
                    onSuccess = { rpcResponse ->
                        if (rpcResponse is WalletApiRpcResponse.SolanaSignMessage) {
                            Result.success(
                                SolanaSignMessageResponse(
                                    signature = rpcResponse.data.signature
                                )
                            )
                        } else {
                            Result.failure(EmbeddedWalletException("Unexpected response type for signMessage"))
                        }
                    },
                    onFailure = { Result.failure(it) }
                )
            } ?: Result.failure(EmbeddedWalletException("Wallet ID is required"))
        } else {
            // Handle other types of SolanaRpcRequest or return an error if only SignMessageParams is supported here
            return Result.failure(EmbeddedWalletException("Unsupported SolanaRpcRequest type for performSolanaRpc. Expected SignMessageParams, got ${params::class.simpleName}"))
        }
    }

    /**
     * Helper function to check if a string is hex encoded (starting with 0x)
     */
    private fun isHexEncoded(value: String): Boolean =
        value.startsWith("0x") && value.drop(2)
            .all { it.isDigit() || it in 'a'..'f' || it in 'A'..'F' }
}