package io.privy.wallet.walletApi.rpc.ethereum

import kotlinx.serialization.KSerializer
import kotlinx.serialization.Serializable
import kotlinx.serialization.descriptors.PrimitiveKind
import kotlinx.serialization.descriptors.PrimitiveSerialDescriptor
import kotlinx.serialization.descriptors.SerialDescriptor
import kotlinx.serialization.encoding.Decoder
import kotlinx.serialization.encoding.Encoder
import kotlinx.serialization.json.JsonDecoder
import kotlinx.serialization.json.intOrNull
import kotlinx.serialization.json.jsonPrimitive

/** Ethereum wallet address represented as a hexadecimal string */
internal typealias EthereumWalletAddress = String

/**
 * ETH "quantity" properties, represented as a hex string, while supporting decoding from an Int value.
 */
@Serializable(with = QuantitySerializer::class)
public class Quantity(public val hexString: String) {
    /**
     * Convert the hexadecimal string to an integer
     */
    public val number: Int?
        get() = if (hexString.startsWith("0x")) {
            hexString.substring(2).toIntOrNull(16)
        } else {
            hexString.toIntOrNull(16)
        }

    public companion object {
        /**
         * Create a Quantity from an integer
         */
        public fun fromNumber(number: Int): Quantity {
            return Quantity("0x${number.toString(16).uppercase()}")
        }
    }
}

/**
 * Custom serializer for Quantity to handle both string and int formats
 */
internal object QuantitySerializer : KSerializer<Quantity> {
    override val descriptor: SerialDescriptor = PrimitiveSerialDescriptor("Quantity", PrimitiveKind.STRING)

    override fun serialize(encoder: Encoder, value: Quantity) {
        encoder.encodeString(value.hexString)
    }

    override fun deserialize(decoder: Decoder): Quantity {
        // Handle both int and string inputs
        if (decoder is JsonDecoder) {
            val element = decoder.decodeJsonElement().jsonPrimitive
            val intValue = element.intOrNull
            if (intValue != null) {
                return Quantity.fromNumber(intValue)
            }
            val stringValue = element.content
            if (isHexEncoded(stringValue)) {
                return Quantity(stringValue)
            }
            throw IllegalArgumentException("Unable to decode as a Quantity")
        } else {
            // Fallback to string-only decoding for non-JSON formats
            return Quantity(decoder.decodeString())
        }
    }

    private fun isHexEncoded(value: String): Boolean {
        return value.startsWith("0x") && value.substring(2).all { 
            it.isDigit() || it in 'a'..'f' || it in 'A'..'F' 
        }
    }
}

/** The transaction type as per EIP-2718 */
internal typealias EIP2718TransactionType = Int

/**
 * Representation of an unsigned Ethereum transaction
 */
@Serializable
public data class UnsignedEthereumTransaction(
    /** The address the transaction is sent from. Must be hexadecimal formatted. */
    val from: EthereumWalletAddress? = null,

    /** Destination address of the transaction. */
    val to: EthereumWalletAddress? = null,

    /** (optional) The nonce to be used for the transaction (hexadecimal or number). */
    val nonce: Quantity? = null,

    /** (optional) The max units of gas that can be used by this transaction (hexadecimal or number). */
    val gasLimit: Quantity? = null,

    /** (optional) The price (in wei) per unit of gas for this transaction (hexadecimal or number), for use in non EIP-1559 transactions (type 0 or 1). */
    val gasPrice: Quantity? = null,

    /** Data to send to the receiving address, especially when calling smart contracts. Must be hexadecimal formatted. */
    val data: String? = null,

    /** (optional) The value (in wei) be sent with the transaction (hexadecimal or number). */
    val value: Quantity? = null,

    /** The chain ID of network your transaction will be sent on (hexadecimal or number). */
    val chainId: Quantity? = null,

    /** (optional) The EIP-2718 transction type (e.g. `2` for EIP-1559 transactions). */
    val type: EIP2718TransactionType? = null,

    /** (optional) The maxFeePerGas (hexadecimal or number) to be used in this transaction, for use in EIP-1559 (type 2) transactions. */
    val maxFeePerGas: Quantity? = null,

    /** (optional) The maxPriorityFeePerGas (hexadecimal or number) to be used in this transaction, for use in EIP-1559 (type 2) transactions. */
    val maxPriorityFeePerGas: Quantity? = null
)

internal fun UnsignedEthereumTransaction.toWalletApi(): WalletApiUnsignedEthereumTransaction {
    return WalletApiUnsignedEthereumTransaction(
        from = this.from,
        to = this.to,
        nonce = this.nonce?.hexString,
        gasLimit = this.gasLimit?.hexString,
        gasPrice = this.gasPrice?.hexString,
        data = this.data,
        value = this.value?.hexString,
        chainId = this.chainId?.number,
        type = this.type,
        maxFeePerGas = this.maxFeePerGas?.hexString,
        maxPriorityFeePerGas = this.maxPriorityFeePerGas?.hexString
    )
}
