package io.privy.wallet.ethereum

import io.privy.auth.EmbeddedWalletException
import io.privy.logging.PrivyLogger
import io.privy.wallet.EmbeddedWalletDetails
import io.privy.wallet.EmbeddedWalletManager
import me.tatarka.inject.annotations.Assisted
import me.tatarka.inject.annotations.Inject

/** The Ethereum wallet's RPC provider */
@Inject
public class RealEmbeddedEthereumWalletProvider(
    @Assisted private val embeddedWalletDetails: EmbeddedWalletDetails,
    private val embeddedWalletManager: EmbeddedWalletManager,
    private val privyLogger: PrivyLogger,
) : EmbeddedEthereumWalletProvider {
  private val walletRpcMethods =
      listOf(
          "eth_sign",
          "personal_sign",
          "secp256k1_sign",
          "eth_populateTransactionRequest",
          "eth_signTypedData_v4",
          "eth_signTransaction",
          "eth_sendTransaction",
      )

  override suspend fun request(request: EthereumRpcRequest): Result<EthereumRpcResponse> {
    privyLogger.internal("Received RPC request! method: ${request.method}")

    return when (request.method) {
      in walletRpcMethods -> {
        embeddedWalletManager.ethereumRpc(
          embeddedWalletDetails = embeddedWalletDetails,
          request = request,
        )
      }
      else -> handleJsonRpc(request)
    }
  }

  private fun handleJsonRpc(request: EthereumRpcRequest): Result<EthereumRpcResponse> {
    privyLogger.error("Unsupported rpc request type")

    return Result.failure(EmbeddedWalletException("Unsupported rpc request type"))
  }
}
