package io.privy.wallet.walletApi

import kotlinx.serialization.KSerializer
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.JsonArray
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonNull
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive
import kotlinx.serialization.json.doubleOrNull
import me.tatarka.inject.annotations.Inject


/**
 * Implementation of [JsonCanonicalizer] that canonicalizes JSON objects according to RFC8785.
 *
 * This involves manually sorting object keys lexicographically.
 *
 * Characteristics and potential deviations from strict canonical forms like RFC 8785:
 * - Key Sorting: Keys are sorted using standard string lexicographical comparison. Strict RFC 8785 requires UTF-8 byte-based sorting.
 * - Slash Escaping: kotlinx.serialization escapes solidus ('/') to ('\/') by default during encoding. Disabling this is not a standard configuration option.
 * - Number Handling: Relies on kotlinx.serialization's default number formatting. Validation for non-finite numbers (NaN/Infinity) is included as they are disallowed in standard JSON and RFC 8785.
 * - This implementation prioritizes a straightforward approach using kotlinx.serialization's features where possible.
 */
@Inject
public class RealJsonCanonicalizer : JsonCanonicalizer {
    // Json configuration used for converting to JsonElement and final encoding.
    private val json: Json = Json {
        prettyPrint = false // Ensures compact output
        isLenient = false // Strict parsing of input
        encodeDefaults = false 
        allowSpecialFloatingPointValues = false // Disallows NaN/Infinity
        // Key sorting and specific escaping rules are handled outside of this configuration.
    }

    /**
     * Encodes a serializable value into a canonical JSON ByteArray using an explicit serializer.
     *
     * @param T The type of the value to encode.
     * @param serializer The KSerializer<T> for the type T.
     * @param value The value to encode.
     * @return A Result containing the canonical JSON representation encoded in UTF-8 as a ByteArray,
     *         or a failure with SerializationException or IllegalArgumentException.
     */
    public override fun <T> encode(serializer: KSerializer<T>, value: T): Result<ByteArray> {
        return runCatching {
            // 1. Serialize to JsonElement
            val jsonElement = json.encodeToJsonElement(serializer, value)
            // 2. Canonicalize
            val canonicalElement = canonicalizeElement(jsonElement)
            // 3. Serialize back to String
            val jsonString = json.encodeToString(JsonElement.serializer(), canonicalElement)
            // 4. Convert to ByteArray using UTF-8
            jsonString.encodeToByteArray()
        }
    }

    /**
     * Recursively transforms a JsonElement into a canonical form.
     * - Sorts object keys lexicographically.
     * - Processes array elements recursively, preserving their order.
     * - Validates that numbers are finite.
     * - Returns primitives and null as is.
     */
    private fun canonicalizeElement(element: JsonElement): JsonElement {
        return when (element) {
            is JsonObject -> {
                // Get the original map of keys to JsonElements.
                val originalMap = element.toMap()

                // Sort keys using standard string lexicographical comparison.
                val sortedKeys = originalMap.keys.sorted()

                // Build a new map with sorted keys and canonicalized values.
                val sortedContent = LinkedHashMap<String, JsonElement>(sortedKeys.size)
                sortedKeys.forEach { key ->
                    sortedContent[key] = canonicalizeElement(originalMap[key]!!)
                }
                JsonObject(sortedContent)
            }
            is JsonArray -> {
                // Process each element in the array recursively.
                JsonArray(element.map { canonicalizeElement(it) })
            }
            is JsonPrimitive -> {
                // Check for non-finite numbers.
                if (!element.isString && element !== JsonNull) {
                    val doubleVal = element.doubleOrNull
                    if (doubleVal != null && !doubleVal.isFinite()) {
                         throw IllegalArgumentException(
                            "Non-finite numbers (NaN/Infinity) disallowed: $doubleVal in $element"
                         )
                    }
                }
                // Return primitives as they are.
                element
            }
        }
    }
}