package io.privy.wallet.walletApi.rpc

import io.privy.wallet.rpc.SolanaSignMessageParams
import io.privy.wallet.walletApi.rpc.ethereum.WalletApiCAIP2
import io.privy.wallet.walletApi.rpc.ethereum.WalletApiEthereumPersonalSignRpcParams
import io.privy.wallet.walletApi.rpc.ethereum.WalletApiEthereumSecp256k1SignRpcParams
import io.privy.wallet.walletApi.rpc.ethereum.WalletApiEthereumSendTransactionRpcParams
import io.privy.wallet.walletApi.rpc.ethereum.WalletApiEthereumSignTransactionRpcParams
import io.privy.wallet.walletApi.rpc.ethereum.WalletApiEthereumSignTypedDataV4RpcParams
import io.privy.wallet.walletApi.rpc.solana.WalletApiSolanaSignMessageRpcParams
import kotlinx.serialization.ExperimentalSerializationApi
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable
import kotlinx.serialization.Transient
import kotlinx.serialization.json.JsonClassDiscriminator

/**
 * Request for wallet RPC operations.
 * This sealed class uses a "method" field as a discriminator to determine the specific request type.
 * The serialized JSON structure will be:
 * {
 * "method": "specific_method_name", // From @SerialName on subclasses
 * "params": { ...parameters_object... } // From the property annotated with @SerialName("params") in subclasses
 * }
 */
@OptIn(ExperimentalSerializationApi::class)
@Serializable
@JsonClassDiscriminator("method")
public sealed class WalletApiRpcRequest {
    /**
     * Abstract property to polymorphically access the parameters object.
     * This is primarily for Kotlin-side convenience and is marked @Transient
     * as the actual serialization of parameters is handled by the specifically
     * annotated field (e.g., `ethereumParams`, `solanaParams`) in each subclass.
     */
    @Transient
    public abstract val params: Any // You could use a more specific common sealed interface for all params if desired

    /**
     * Ethereum personal_sign request.
     */
    @Serializable
    @SerialName("personal_sign") // Value for the "method" discriminator field
    public data class EthereumPersonalSign(
        /**
         * The parameters for this specific RPC method.
         * This object will be serialized under the "params" key in the JSON.
         */
        @SerialName("params")
        val ethereumParams: WalletApiEthereumPersonalSignRpcParams
    ) : WalletApiRpcRequest() {
        @Transient // This override is for the polymorphic 'params' accessor
        override val params: WalletApiEthereumPersonalSignRpcParams = ethereumParams
    }

    /**
     * Solana signMessage request.
     */
    @Serializable
    @SerialName("signMessage") // Value for the "method" discriminator field
    public data class SolanaSignMessage(
        /**
         * The parameters for this specific RPC method.
         * This object will be serialized under the "params" key in the JSON.
         */
        @SerialName("params")
        val solanaParams: WalletApiSolanaSignMessageRpcParams
    ) : WalletApiRpcRequest() {
        @Transient // This override is for the polymorphic 'params' accessor
        override val params: WalletApiSolanaSignMessageRpcParams = solanaParams
    }
    
    /**
     * Ethereum secp256k1_sign request.
     */
    @Serializable
    @SerialName("secp256k1_sign") // Value for the "method" discriminator field
    public data class EthereumSecp256k1Sign(
        /**
         * The parameters for this specific RPC method.
         * This object will be serialized under the "params" key in the JSON.
         */
        @SerialName("params")
        val ethereumParams: WalletApiEthereumSecp256k1SignRpcParams
    ) : WalletApiRpcRequest() {
        @Transient // This override is for the polymorphic 'params' accessor
        override val params: WalletApiEthereumSecp256k1SignRpcParams = ethereumParams
    }
    /**
     * Ethereum eth_signTypedDataV4 request.
     */
    @Serializable
    @SerialName("eth_signTypedData_v4") // Value for the "method" discriminator field
    public data class EthereumSignTypedDataV4(
        /**
         * The parameters for this specific RPC method.
         * This object will be serialized under the "params" key in the JSON.
         */
        @SerialName("params")
        val ethereumParams: WalletApiEthereumSignTypedDataV4RpcParams
    ) : WalletApiRpcRequest() {
        @Transient // This override is for the polymorphic 'params' accessor
        override val params: WalletApiEthereumSignTypedDataV4RpcParams = ethereumParams
    }
    
    /**
     * Ethereum eth_signTransaction request.
     */
    @Serializable
    @SerialName("eth_signTransaction") // Value for the "method" discriminator field
    public data class EthereumSignTransaction(
        /**
         * The parameters for this specific RPC method.
         * This object will be serialized under the "params" key in the JSON.
         */
        @SerialName("params")
        val ethereumParams: WalletApiEthereumSignTransactionRpcParams
    ) : WalletApiRpcRequest() {
        @Transient // This override is for the polymorphic 'params' accessor
        override val params: WalletApiEthereumSignTransactionRpcParams = ethereumParams
    }
    
    /**
     * Ethereum eth_sendTransaction request.
     */
    @Serializable
    @SerialName("eth_sendTransaction") // Value for the "method" discriminator field
    public data class EthereumSendTransaction(
        /**
         * The parameters for this specific RPC method.
         * This object will be serialized under the "params" key in the JSON.
         */
        @SerialName("params")
        val ethereumParams: WalletApiEthereumSendTransactionRpcParams,
        
        /**
         * The CAIP-2 chain ID for the transaction.
         * This is required to properly route the transaction to the correct chain.
         */
        @SerialName("caip2")
        val caip2: WalletApiCAIP2
    ) : WalletApiRpcRequest() {
        @Transient // This override is for the polymorphic 'params' accessor
        override val params: WalletApiEthereumSendTransactionRpcParams = ethereumParams
    }
}