package io.privy.wallet.walletApi.rpc

import io.privy.wallet.walletApi.rpc.ethereum.WalletApiEthereumPersonalSignRpcResponse
import io.privy.wallet.walletApi.rpc.ethereum.WalletApiEthereumSecp256k1SignRpcResponse
import io.privy.wallet.walletApi.rpc.ethereum.WalletApiEthereumSendTransactionRpcResponseData
import io.privy.wallet.walletApi.rpc.ethereum.WalletApiEthereumSignTransactionRpcResponseData
import io.privy.wallet.walletApi.rpc.ethereum.WalletApiEthereumSignTypedDataV4RpcResponse
import io.privy.wallet.walletApi.rpc.solana.WalletApiSolanaSignMessageRpcResponse
import kotlinx.serialization.ExperimentalSerializationApi
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.JsonClassDiscriminator

/**
 * Generic wallet RPC response.
 * This sealed class uses a "method" field as a discriminator to determine the specific response type.
 * The deserialized JSON structure is expected to be:
 * {
 * "method": "specific_method_name", // From @SerialName on subclasses
 * "data": { ...response_data_object... } // The 'data' property in subclasses maps to this
 * }
 */
@OptIn(ExperimentalSerializationApi::class)
@Serializable
@JsonClassDiscriminator("method")
public sealed class WalletApiRpcResponse {

    /**
     * Ethereum personal_sign response.
     */
    @Serializable
    @SerialName("personal_sign") // Value for the "method" discriminator field
    public data class EthereumPersonalSign(
        /**
         * Response data containing the wallet-produced signature.
         * This object is expected under the "data" key in the JSON.
         */
        val data: WalletApiEthereumPersonalSignRpcResponse
    ) : WalletApiRpcResponse()

    /**
     * Solana signMessage response.
     */
    @Serializable
    @SerialName("signMessage") // Value for the "method" discriminator field
    public data class SolanaSignMessage(
        /**
         * Response data containing the wallet-produced signature.
         * This object is expected under the "data" key in the JSON.
         */
        val data: WalletApiSolanaSignMessageRpcResponse
    ) : WalletApiRpcResponse()
    
    /**
     * Ethereum secp256k1_sign response.
     */
    @Serializable
    @SerialName("secp256k1_sign") // Value for the "method" discriminator field
    public data class EthereumSecp256k1Sign(
        /**
         * Response data containing the wallet-produced signature.
         * This object is expected under the "data" key in the JSON.
         */
        val data: WalletApiEthereumSecp256k1SignRpcResponse
    ) : WalletApiRpcResponse()
    
    /**
     * Ethereum eth_signTypedDataV4 response.
     */
    @Serializable
    @SerialName("eth_signTypedData_v4") // Value for the "method" discriminator field
    public data class EthereumSignTypedDataV4(
        /**
         * Response data containing the wallet-produced signature.
         * This object is expected under the "data" key in the JSON.
         */
        val data: WalletApiEthereumSignTypedDataV4RpcResponse
    ) : WalletApiRpcResponse()
    
    /**
     * Ethereum eth_signTransaction response.
     */
    @Serializable
    @SerialName("eth_signTransaction") // Value for the "method" discriminator field
    public data class EthereumSignTransaction(
        /**
         * Response data containing the wallet-produced signed transaction.
         * This object is expected under the "data" key in the JSON.
         */
        val data: WalletApiEthereumSignTransactionRpcResponseData
    ) : WalletApiRpcResponse()
    
    /**
     * Ethereum eth_sendTransaction response.
     */
    @Serializable
    @SerialName("eth_sendTransaction") // Value for the "method" discriminator field
    public data class EthereumSendTransaction(
        /**
         * Response data containing details about the sent transaction.
         * This object is expected under the "data" key in the JSON.
         */
        val data: WalletApiEthereumSendTransactionRpcResponseData
    ) : WalletApiRpcResponse()
}