package io.privy.wallet.webview

import io.privy.auth.AuthStateRepository
import io.privy.auth.EmbeddedWalletException
import io.privy.auth.internal.InternalAuthSession
import io.privy.auth.internal.InternalAuthState
import io.privy.auth.internal.InternalLinkedAccount
import io.privy.auth.internal.primaryWalletForEntropyInfo
import io.privy.auth.internal.privyStack
import io.privy.auth.internal.toEmbeddedWalletDetails
import io.privy.logging.PrivyLogger
import io.privy.wallet.PrivyStack
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.flow.filter
import kotlinx.coroutines.launch
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import me.tatarka.inject.annotations.Inject

public interface WebViewWalletConnector {
  /**
   * Helper function that first checks a user is authenticated
   * Then checks if the wallet is connected
   * If it isn't, it attempts to connect the wallet
   */
  public suspend fun awaitConnected(): Result<AuthenticatedAndConnectedState>

  /**
   * Proactively attempt to connect a users wallet in the background
   */
  public fun connectWalletInBackgroundIfNeeded()
}

@Inject
public class RealWebViewWalletConnector(
  private val webViewHandler: WebViewHandler,
  private val privyLogger: PrivyLogger,
  private val authStateRepository: AuthStateRepository,
): WebViewWalletConnector {
  private val scope = CoroutineScope(SupervisorJob() + Dispatchers.Default)

  private var embeddedWalletState =
    MutableStateFlow<EmbeddedWalletState>(EmbeddedWalletState.Disconnected)

  private val connectWalletMutex = Mutex()

  init {
    subscribeToAuthStateUpdates()
  }

  override suspend fun awaitConnected(): Result<AuthenticatedAndConnectedState> {
    ensureAuthenticated()
      .fold(
        onSuccess = { internalAuthSession ->
          return internalAuthSession.attemptConnectWallet().map { walletAddress ->
            AuthenticatedAndConnectedState(
              accessToken = internalAuthSession.accessToken, walletAddress = walletAddress)
          }
        },
        onFailure = {
          return Result.failure(it)
        })
  }

  override fun connectWalletInBackgroundIfNeeded() {
    scope.launch {
      val internalAuthState = authStateRepository.internalAuthState.value

      if (internalAuthState is InternalAuthState.Authenticated) {
        // if user authenticates, and they have embedded wallets, we should connect the wallet if
        // needed. this call internally throws a failure if wallet is not a legacy wallet,
        // and we fail silently (rightfully so)
        internalAuthState.session.attemptConnectWallet()
      }
    }
  }

  // Helper function to ensure user is authenticated before making wallet call
  // If authenticated: returns success with associated accessToken
  // If unauthenticated: returns result failure
  private fun ensureAuthenticated(): Result<InternalAuthSession> {
    val authState = authStateRepository.internalAuthState.value

    return if (authState is InternalAuthState.Authenticated) {
      Result.success(authState.session)
    } else {
      Result.failure(exception = EmbeddedWalletException("User is not authenticated."))
    }
  }


  private suspend fun InternalAuthSession.attemptConnectWallet(): Result<String> {
    val primaryEmbeddedWallet =
      this.user.linkedAccounts.primaryWalletForEntropyInfo()
        ?: return Result.failure(
          EmbeddedWalletException("User doesn't have an embedded wallet."))

    // Only legacy wallets need to be connected via webview
    if (primaryEmbeddedWallet.privyStack != PrivyStack.Legacy) {
      privyLogger.internal("This wallet type does not need to be connected via webview.")

      return Result.failure(
        EmbeddedWalletException("Wallet type does not need to be connected via webview.")
      )
    }

    val embeddedWalletState = embeddedWalletState.value
    return if (embeddedWalletState is EmbeddedWalletState.Connected) {
      // User is already connected, return success
      Result.success(embeddedWalletState.walletAddress)
    } else {
      privyLogger.error("Embedded wallet exists and is not connected, attempting to connect.")
      connectWallet(accessToken = this.accessToken, primaryEmbeddedWallet = primaryEmbeddedWallet)
    }
  }

  private suspend fun connectWallet(
    accessToken: String,
    primaryEmbeddedWallet: InternalLinkedAccount.EmbeddedWalletAccount,
  ): Result<String> {
    val walletState = embeddedWalletState.value
    if (walletState is EmbeddedWalletState.Connected) {
      // If wallet is already connected, no op
      privyLogger.info("Wallet already connected, no op.")
      return Result.success(walletState.walletAddress)
    }

    // Wrap this in a mutex so that we're only attempting to connect one at a time
    return connectWalletMutex.withLock {
      connectWallet(
        accessToken = accessToken,
        primaryEmbeddedWallet = primaryEmbeddedWallet,
        attemptRecovery = true,
      )
    }
  }

  // Most callers should not call this directly, and should call
  // connectWallet(accessToken, primaryWalletAddress) directly, since that function internally
  // handle
  // mutual exclusion
  private suspend fun connectWallet(
    accessToken: String,
    primaryEmbeddedWallet: InternalLinkedAccount.EmbeddedWalletAccount,
    attemptRecovery: Boolean
  ): Result<String> {
    // Check again, in case a prior caller resulted in connected state
    val walletState = embeddedWalletState.value
    if (walletState is EmbeddedWalletState.Connected) {
      // If wallet is already connected, no op
      privyLogger.info("Wallet already connected, no op.")
      return Result.success(walletState.walletAddress)
    }

    val connectWalletResponse =
      webViewHandler.connectWallet(
        accessToken = accessToken,
        embeddedWalletDetails =
        primaryEmbeddedWallet.toEmbeddedWalletDetails(
          primaryEmbeddedWallet = primaryEmbeddedWallet),
      )

    return when (connectWalletResponse) {
      is IFrameResponse.Error -> {
        if (connectWalletResponse.type == "wallet_not_on_device" && attemptRecovery) {
          privyLogger.info("Wallet needs recovery. Attempting to recover.")
          // Wallet needs recovery
          recoverWalletThenTryConnecting(
            accessToken = accessToken, primaryEmbeddedWallet = primaryEmbeddedWallet)
        } else {
          // Some other error - surface it as failure
          val errorMessage = connectWalletResponse.message
          privyLogger.error("Error connecting wallet: $errorMessage")
          Result.failure(EmbeddedWalletException(errorMessage))
        }
      }
      is IFrameResponse.Success -> {
        privyLogger.info("Wallet '${primaryEmbeddedWallet.address}' connected!")
        // connect success! update state to connected
        val connectedWalletAddress = connectWalletResponse.value.data.entropyId
        embeddedWalletState.value = EmbeddedWalletState.Connected(connectedWalletAddress)

        Result.success(connectedWalletAddress)
      }
    }
  }

  private fun subscribeToAuthStateUpdates() {
    scope.launch {
      authStateRepository
        .internalAuthState
        .collectLatest { internalAuthState ->
          if (internalAuthState is InternalAuthState.Authenticated) {
            // Proactively connect wallet if needed when user is authenticated
            connectWalletInBackgroundIfNeeded()
          } else if (internalAuthState is InternalAuthState.Unauthenticated) {
            // If user logs out, reset embedded wallet state
            privyLogger.internal("User is unauthenticated. Disconnecting wallet.")
            embeddedWalletState.value = EmbeddedWalletState.Disconnected
          }
        }
    }
  }

  private suspend fun recoverWalletThenTryConnecting(
    accessToken: String,
    primaryEmbeddedWallet: InternalLinkedAccount.EmbeddedWalletAccount,
  ): Result<String> {
    // try recovering
    val recoverWalletResponse =
      webViewHandler.recoverWallet(
        accessToken = accessToken,
        embeddedWalletDetails =
        primaryEmbeddedWallet.toEmbeddedWalletDetails(
          primaryEmbeddedWallet = primaryEmbeddedWallet))

    // if success, try connecting with attemptRecovery == false
    return when (recoverWalletResponse) {
      is IFrameResponse.Error -> {
        // Recovery error, so just return the error up.
        val errorMessage = recoverWalletResponse.message
        privyLogger.error("Error recovering wallet: $errorMessage")
        Result.failure(EmbeddedWalletException(errorMessage))
      }
      is IFrameResponse.Success -> {
        privyLogger.info("Successfully recovered wallet. Attempting to connect wallet now.")
        connectWallet(
          accessToken = accessToken,
          primaryEmbeddedWallet = primaryEmbeddedWallet,
          attemptRecovery = false // important to set this to false to prevent infinite recursion
        )
      }
    }
  }
}