package io.privy.wallet

import io.privy.appconfig.AppConfigRepository
import io.privy.auth.PrivyAuth
import io.privy.auth.internal.InternalAuthStateRepository
import io.privy.auth.flatMap
import io.privy.auth.internal.InternalAuthState
import io.privy.auth.internal.embeddedEthereumWalletAccounts
import io.privy.auth.internal.embeddedSolanaWalletAccounts
import io.privy.auth.internal.primaryEthereumWalletAccountOrNull
import io.privy.auth.internal.primarySolanaWalletAccountOrNull
import io.privy.auth.internal.primaryWalletForEntropyInfo
import io.privy.auth.internal.toEntropyWalletDetails
import io.privy.logging.PrivyLogger
import io.privy.wallet.creation.WalletCreatorFactory
import io.privy.wallet.ethereum.EthereumRpcRequest
import io.privy.wallet.ethereum.EthereumRpcResponse
import io.privy.wallet.rpc.WalletRpcExecutorFactory
import io.privy.wallet.solana.SolanaRpcRequest
import io.privy.wallet.solana.SolanaSignMessageResponse
import io.privy.wallet.webview.WebViewWalletConnector
import me.tatarka.inject.annotations.Inject

// The layer between KMP SDK and WebViewHandler
// In charge of managing web view states so that it is abstracted to callers
@Inject
public class RealEmbeddedWalletManager(
  private val privyLogger: PrivyLogger,
  private val internalAuthStateRepository: InternalAuthStateRepository,
  // Ensure this is always initialized here to proatively kick off webview processes
  private val webViewWalletConnector: WebViewWalletConnector,
  private val rpcExecutorFactory: WalletRpcExecutorFactory,
  // Inject lazily to prevent circular dependency
  private val lazyPrivyAuth: Lazy<PrivyAuth>,
  private val appConfigRepository: AppConfigRepository,
  private val walletCreatorFactory: WalletCreatorFactory,
) : EmbeddedWalletManager {
  private val privyAuth by lazyPrivyAuth

  override suspend fun createEthereumWallet(
    allowAdditional: Boolean
  ): Result<String> {
    return privyAuth.ensureAuthenticated { authSession ->
      val accessToken = authSession.accessToken

      val ethereumWalletsCount = authSession.user.linkedAccounts.embeddedEthereumWalletAccounts().size

      if (ethereumWalletsCount == 0) {
        // User does not have an embedded wallet yet, need to create primary wallet
        val primarySolanaWalletAddress = authSession.user.linkedAccounts.primarySolanaWalletAccountOrNull()

        createPrimaryEthereumWallet(
          accessToken = accessToken,
          primarySolanaWalletAddress = primarySolanaWalletAddress?.address,
        )
      } else {
        // We know this should never be null, but return failure in rare case it is.
        val primaryWalletForEntropy = authSession.user.linkedAccounts.primaryWalletForEntropyInfo()
          ?: return@ensureAuthenticated Result.failure(EmbeddedWalletException("Primary wallet for entropy is null"))

        // User already has a primary wallet, create additional if allowed
        createAdditionalEthereumWalletIfAllowed(
          allowAdditional = allowAdditional,
          accessToken = accessToken,
          currentWalletCount = ethereumWalletsCount,
          entropyWalletDetails = primaryWalletForEntropy.toEntropyWalletDetails()
        )
      }
    }
  }

  override suspend fun createSolanaWallet(
    allowAdditional: Boolean,
  ): Result<String> {
    return privyAuth.ensureAuthenticated { authSession ->
      val accessToken = authSession.accessToken

      val solanaWalletsCount = authSession.user.linkedAccounts.embeddedSolanaWalletAccounts().size

      if (solanaWalletsCount == 0) {
        // User does not have a Solana embedded wallet yet, need to create primary wallet
        val primaryEthereumWallet = authSession.user.linkedAccounts.primaryEthereumWalletAccountOrNull()

        createPrimarySolanaWallet(
          accessToken = accessToken,
          primaryEmbeddedEthereumWalletAddress = primaryEthereumWallet?.address,
        )
      } else {
        // We know this should never be null, but return failure in rare case it is.
        val primaryWalletForEntropy = authSession.user.linkedAccounts.primaryWalletForEntropyInfo()
          ?: return@ensureAuthenticated Result.failure(EmbeddedWalletException("Primary wallet for entropy is null"))
        
        createAdditionalSolanaWalletIfAllowed(
          accessToken = accessToken,
          allowAdditional = allowAdditional,
          currentWalletCount = solanaWalletsCount,
          entropyWalletDetails = primaryWalletForEntropy.toEntropyWalletDetails()
        )
      }
    }
  }

  /**
   * Internal method in charge of creating the user's / first Ethereum primary wallet (hdIndex == 0)
   * Caller of this method should verify user is authenticated and pass in a valid access token
   */
  private suspend fun createPrimaryEthereumWallet(
    accessToken: String,
    primarySolanaWalletAddress: String?,
  ): Result<String> {
    return appConfigRepository.loadAppConfig()
      .flatMap { appConfig ->
        walletCreatorFactory.create(appConfig.embeddedWalletConfig)
          .createEthereumWallet(
            accessToken = accessToken,
            existingSolanaWalletAddress = primarySolanaWalletAddress,
          )
      }
  }

  /**
   * Internal method in charge of creating the user's / first Solana primary wallet (hdIndex == 0)
   * Caller of this method should verify user is authenticated and pass in a valid access token
   */
  private suspend fun createPrimarySolanaWallet(
    accessToken: String,
    primaryEmbeddedEthereumWalletAddress: String?,
  ): Result<String> {
    return appConfigRepository.loadAppConfig()
      .flatMap { appConfig ->
        walletCreatorFactory.create(appConfig.embeddedWalletConfig)
          .createSolanaWallet(
            accessToken = accessToken,
            existingEthereumWalletAddress = primaryEmbeddedEthereumWalletAddress,
          )
      }
  }

  /**
   * Internal method in charge of creating an additional wallet for the user's if dev specifies
   * additional wallets are allowed. Caller of this method should verify user is authenticated and
   * pass in a valid access token, and should verify a user already has a primary embedded wallet
   * account.
   */
  private suspend fun createAdditionalEthereumWalletIfAllowed(
    allowAdditional: Boolean,
    accessToken: String,
    currentWalletCount: Int,
    entropyWalletDetails: EmbeddedWalletDetails.EntropyWalletDetails
  ): Result<String> {
    return if (allowAdditional) {
      createAdditionalWallet(
        accessToken = accessToken,
        currentWalletCount = currentWalletCount,
        entropyWalletDetails = entropyWalletDetails
      )
    } else {
      // Dev didn't specify allowAdditional == true, so throw an error
      val errorMessage =
        "User already has an embedded wallet. To create an additional wallet, set allowAdditional to true."
      privyLogger.error(errorMessage)

      Result.failure(exception = EmbeddedWalletException(message = errorMessage))
    }
  }

  /**
   * Internal method in charge of creating an additional wallet for the user's if dev specifies
   * additional wallets are allowed. Caller of this method should verify user is authenticated and
   * pass in a valid access token, and should verify a user already has a primary embedded wallet
   * account.
   */
  private suspend fun createAdditionalSolanaWalletIfAllowed(
    allowAdditional: Boolean,
    accessToken: String,
    currentWalletCount: Int,
    entropyWalletDetails: EmbeddedWalletDetails.EntropyWalletDetails
  ): Result<String> {
    return if (allowAdditional) {
      createAdditionalSolanaWallet(
        accessToken = accessToken,
        currentWalletCount = currentWalletCount,
        entropyWalletDetails = entropyWalletDetails
      )
    } else {
      // Dev didn't specify allowAdditional == true, so throw an error
      val errorMessage =
        "User already has an embedded wallet. To create an additional wallet, set allowAdditional to true."
      privyLogger.error(errorMessage)

      Result.failure(exception = EmbeddedWalletException(message = errorMessage))
    }
  }

  /**
   * Internal method in charge of creating an additional wallet for the user's (hdIndex > 0) Caller
   * of this method should verify user is authenticated and pass in a valid access token, and should
   * verify a user already has a primary embedded wallet account.
   */
  private suspend fun createAdditionalWallet(
    accessToken: String,
    currentWalletCount: Int,
    entropyWalletDetails: EmbeddedWalletDetails.EntropyWalletDetails
  ): Result<String> {
    return appConfigRepository.loadAppConfig()
      .flatMap { appConfig ->
        walletCreatorFactory.create(appConfig.embeddedWalletConfig)
          .createAdditionalEthereumWallet(
            accessToken = accessToken,
            // The new wallets HD index should be the current highest wallet's index + 1
            // which is the same as the total count of embedded wallets
            hdWalletIndex = currentWalletCount,
            entropyWalletDetails = entropyWalletDetails
          )
      }
  }
  /**
   * Internal method in charge of creating an additional Solana wallet for the user's (hdIndex > 0) Caller
   * of this method should verify user is authenticated and pass in a valid access token, and should
   * verify a user already has a primary embedded wallet account.
   */
  private suspend fun createAdditionalSolanaWallet(
    accessToken: String,
    currentWalletCount: Int,
    entropyWalletDetails: EmbeddedWalletDetails.EntropyWalletDetails
  ): Result<String> {

    return appConfigRepository.loadAppConfig()
      .flatMap { appConfig ->
        walletCreatorFactory.create(appConfig.embeddedWalletConfig)
          .createAdditionalSolanaWallet(
            accessToken = accessToken,
            hdWalletIndex = currentWalletCount,
            entropyWalletDetails = entropyWalletDetails
          )
      }
  }

  override suspend fun ethereumRpc(
      embeddedWalletDetails: EmbeddedWalletDetails,
      request: EthereumRpcRequest
  ): Result<EthereumRpcResponse> {
    val rpcExecutor = rpcExecutorFactory.create(embeddedWalletDetails)

    // Get authentication token directly from AuthManager for unified stack wallets
    return privyAuth.refreshSessionIfNeeded()
      .map { it.accessToken }
      .flatMap { accessToken ->
        // Execute the request through the RPC executor with the auth token
        rpcExecutor.performEthRpc(
          request = request,
          embeddedWalletDetails = embeddedWalletDetails,
          accessToken = accessToken
        )
      }
  }

  override suspend fun solanaRpc(
      embeddedWalletDetails: EmbeddedWalletDetails,
      request: SolanaRpcRequest
  ): Result<SolanaSignMessageResponse> {
      val rpcExecutor = rpcExecutorFactory.create(embeddedWalletDetails)

      // Get authentication token directly from AuthManager for unified stack wallets
      // The rpcExecutor.performSolanaRpc now expects SolanaRpcRequest directly.
      return privyAuth.refreshSessionIfNeeded()
        .map { it.accessToken }
        .flatMap { accessToken ->
          // Execute the request through the RPC executor with the auth token
          rpcExecutor.performSolanaRpc(
            request = request, // Pass SolanaRpcRequest directly
            embeddedWalletDetails = embeddedWalletDetails,
            accessToken = accessToken
          )
        }
  }

  override fun connectWalletIfNeeded() {
    val authState = internalAuthStateRepository.internalAuthState.value

    if (authState is InternalAuthState.Authenticated) {
      webViewWalletConnector.connectWalletInBackgroundIfNeeded()
    }
  }
}
