package io.privy.wallet.creation

import io.privy.wallet.EmbeddedWalletException
import io.privy.logging.PrivyLogger
import io.privy.wallet.ChainType
import io.privy.wallet.EmbeddedWalletDetails
import io.privy.wallet.webview.IFrameResponse
import io.privy.wallet.webview.WebViewHandler
import me.tatarka.inject.annotations.Inject


private val ChainType.chainTypeName: String
  get() = when(this) {
    ChainType.Ethereum -> "ethereum"
    ChainType.Solana -> "solana"
  }

@Inject
public class RealWebViewWalletCreator(
  private val webViewHandler: WebViewHandler,
  private val privyLogger: PrivyLogger,
): WalletCreator {
  public companion object {
    public const val diQualifierName: String = "WebViewWalletCreator"
  }

  override suspend fun createEthereumWallet(
    accessToken: String,
    existingSolanaWalletAddress: String?,
  ): Result<String> {
    return when (
      val createWalletResponse = webViewHandler.createEthereumWallet(
        accessToken = accessToken, existingSolanaWalletAddress = existingSolanaWalletAddress,
      )
    ) {
      is IFrameResponse.Error -> {
        // Some other error - surface it as failure
        val errorMessage = createWalletResponse.message
        privyLogger.error("Error creating Ethereum wallet: $errorMessage")
        Result.failure(EmbeddedWalletException(errorMessage))
      }
      is IFrameResponse.Success -> {
        val newWalletAddress = createWalletResponse.value.data.address
        privyLogger.debug("Newly created Ethereum wallet address: $newWalletAddress")

        Result.success(newWalletAddress)
      }
    }
  }

  override suspend fun createAdditionalEthereumWallet(
    accessToken: String,
    hdWalletIndex: Int,
    entropyWalletDetails: EmbeddedWalletDetails.EntropyWalletDetails
  ): Result<String> {

    // For Ethereum additional wallets, the entropy always comes from the primary Ethereum wallet
    val response =
      webViewHandler.createAdditionalWallet(
        accessToken = accessToken,
        chainType = ChainType.Ethereum.chainTypeName,
        entropyId = entropyWalletDetails.entropyId,
        entropyIdVerifier = entropyWalletDetails.entropyIdVerifier,
        hdWalletIndex = hdWalletIndex,
      )

    return when (response) {
      is IFrameResponse.Error -> {
        // Some other error - surface it as failure
        val errorMessage = response.message
        privyLogger.error("Error creating additional ethereum wallet: $errorMessage")
        Result.failure(EmbeddedWalletException(errorMessage))
      }
      is IFrameResponse.Success -> {
        val newWalletAddress = response.value.data.address
        privyLogger.debug("Newly created ethereum wallet address: $newWalletAddress")
        Result.success(newWalletAddress)
      }
    }
  }

  override suspend fun createAdditionalSolanaWallet(
    accessToken: String,
    hdWalletIndex: Int,
    entropyWalletDetails: EmbeddedWalletDetails.EntropyWalletDetails
  ): Result<String> {

    val response = webViewHandler.createAdditionalWallet(
      accessToken = accessToken,
      chainType = ChainType.Solana.chainTypeName,
      entropyId = entropyWalletDetails.address,
      entropyIdVerifier = entropyWalletDetails.entropyIdVerifier,
      hdWalletIndex = hdWalletIndex,
    )

    return when (response) {
      is IFrameResponse.Error -> {
        privyLogger.error("Error creating additional solana wallet: ${response.message}")
        Result.failure(EmbeddedWalletException(response.message))
      }
      is IFrameResponse.Success -> {
        val newWalletAddress = response.value.data.address
        privyLogger.debug("Successfully created additional solana wallet with ${entropyWalletDetails.entropyIdVerifier}: $newWalletAddress")
        Result.success(newWalletAddress)
      }
    }
  }

  override suspend fun createSolanaWallet(
    accessToken: String,
    existingEthereumWalletAddress: String?,
  ): Result<String> {
    return when (val createWalletResponse =
      webViewHandler.createSolanaWallet(
        accessToken = accessToken,
        existingEthereumWalletAddress = existingEthereumWalletAddress)) {
      is IFrameResponse.Error -> {
        // Some other error - surface it as failure
        val errorMessage = createWalletResponse.message
        privyLogger.error("Error creating Solana wallet: $errorMessage")
        Result.failure(EmbeddedWalletException(errorMessage))
      }
      is IFrameResponse.Success -> {
        val newWalletAddress = createWalletResponse.value.data.publicKey
        privyLogger.debug("Newly created Solana wallet address: $newWalletAddress")

        Result.success(newWalletAddress)
      }
    }
  }
}