package io.privy.wallet.walletApi.rpc.ethereum

import kotlinx.serialization.KSerializer
import kotlinx.serialization.Serializable
import kotlinx.serialization.descriptors.PrimitiveKind
import kotlinx.serialization.descriptors.PrimitiveSerialDescriptor
import kotlinx.serialization.descriptors.SerialDescriptor
import kotlinx.serialization.encoding.Decoder
import kotlinx.serialization.encoding.Encoder

/**
 * Represents a CAIP-2 chain identifier.
 * CAIP-2 format: namespace:reference (e.g., "eip155:1" for Ethereum mainnet)
 * 
 * See specification: https://github.com/ChainAgnostic/CAIPs/blob/main/CAIPs/caip-2.md
 */
@Serializable(with = WalletApiCAIP2Serializer::class)
public class WalletApiCAIP2(private val caip2String: String) {
    
    // Private constructor to enforce validation through factory methods
    private constructor(namespace: String, reference: String) : this("$namespace:$reference")
    
    /**
     * Creates a CAIP-2 identifier from a raw string.
     * Returns null if the string doesn't match the CAIP-2 format.
     */
    public companion object {
        private val CAIP2_REGEX = Regex("^[-a-z0-9]{3,8}:[-_a-zA-Z0-9]{1,32}$")
        
        /**
         * Create a CAIP-2 identifier from a string if valid, or null if not valid.
         */
        public fun fromString(string: String): WalletApiCAIP2? {
            return if (CAIP2_REGEX.matches(string)) {
                WalletApiCAIP2(string)
            } else {
                null
            }
        }
        
        /**
         * Create a CAIP-2 identifier for an EIP-155 chain (Ethereum and compatible chains).
         */
        public fun forEip155Chain(chainId: Int): WalletApiCAIP2 {
            return WalletApiCAIP2("eip155", chainId.toString())
        }
    }
    
    /**
     * Returns the namespace part of the CAIP-2 identifier (e.g., "eip155").
     */
    public val namespace: String
        get() = caip2String.split(":")[0]
            
    /**
     * Returns the reference part of the CAIP-2 identifier (e.g., "1").
     */
    public val reference: String
        get() = caip2String.split(":")[1]
    
    /**
     * Returns the string representation of this CAIP-2 identifier.
     */
    override fun toString(): String = caip2String
    
    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (other !is WalletApiCAIP2) return false
        return caip2String == other.caip2String
    }
    
    override fun hashCode(): Int {
        return caip2String.hashCode()
    }
}

/**
 * Serializer for WalletApiCAIP2 that encodes/decodes as a simple string.
 */
internal object WalletApiCAIP2Serializer : KSerializer<WalletApiCAIP2> {
    override val descriptor: SerialDescriptor = 
        PrimitiveSerialDescriptor("WalletApiCAIP2", PrimitiveKind.STRING)
    
    override fun serialize(encoder: Encoder, value: WalletApiCAIP2) {
        encoder.encodeString(value.toString())
    }
    
    override fun deserialize(decoder: Decoder): WalletApiCAIP2 {
        val string = decoder.decodeString()
        return WalletApiCAIP2.fromString(string) ?: 
            throw IllegalArgumentException("Chain ID must be a valid CAIP-2 chain ID, e.g. 'eip155:1'")
    }
}
