package io.privy.wallet.webview

import io.privy.wallet.EmbeddedWalletException
import io.privy.auth.flatMap
import io.privy.logging.PrivyLogger
import io.privy.wallet.EmbeddedWalletDetails
import io.privy.wallet.rpc.WalletRpcExecutor
import io.privy.wallet.ethereum.EthereumRpcRequest
import io.privy.wallet.ethereum.EthereumRpcResponse
import io.privy.wallet.solana.SolanaRpcRequest
import io.privy.wallet.solana.SolanaSignMessageResponse
import me.tatarka.inject.annotations.Inject

@Inject
public class RealWebViewWalletRpcExecutor(
  private val webViewWalletConnector: WebViewWalletConnector,
  private val webViewHandler: WebViewHandler,
  private val privyLogger: PrivyLogger,
): WalletRpcExecutor {
  public companion object {
    public const val diQualifierName: String = "RealWebViewWalletRPCExecutor"
  }

  override suspend fun performEthRpc(
    request: EthereumRpcRequest,
    embeddedWalletDetails: EmbeddedWalletDetails,
    accessToken: String
  ): Result<EthereumRpcResponse> {
    return webViewWalletConnector.awaitConnected()
      .flatMap { authenticatedAndConnectedState ->
          val rpcResponse =
            webViewHandler.ethereumRpc(
              accessToken = authenticatedAndConnectedState.accessToken,
              embeddedWalletDetails = embeddedWalletDetails,
              request = request,
            )

          return when (rpcResponse) {
            is IFrameResponse.Success -> {
              privyLogger.info("Wallet RPC response: ${rpcResponse.value.data}")
              Result.success(
                EthereumRpcResponse(
                  method = rpcResponse.value.data.response.method,
                  data = rpcResponse.value.data.response.data))
            }
            is IFrameResponse.Error -> {
              // Some other error - surface it as failure
              val errorMessage = rpcResponse.message
              privyLogger.error("Error sending rpc request: $errorMessage")
              Result.failure(EmbeddedWalletException(errorMessage))
            }
          }
        }
  }

  override suspend fun performSolanaRpc(
    request: SolanaRpcRequest,
    embeddedWalletDetails: EmbeddedWalletDetails,
    accessToken: String
  ): Result<SolanaSignMessageResponse> {
    // For non-unified stack wallets or other Solana RPC methods, use the existing WebView flow
    return webViewWalletConnector.awaitConnected().flatMap { authenticatedAndConnectedState ->
      webViewHandler
        .solanaRpc(
          accessToken = authenticatedAndConnectedState.accessToken,
          embeddedWalletDetails = embeddedWalletDetails,
          request = request,
        )
        .onSuccess { privyLogger.info("Solana RPC response: ${it.signature}") }
        .onFailure {
          privyLogger.error("Error sending Solana RPC request: ${it.message}")
        }
    }
  }
}