package io.privy.wallet.creation

import io.privy.auth.EmbeddedWalletException
import io.privy.logging.PrivyLogger
import io.privy.wallet.webview.IFrameResponse
import io.privy.wallet.webview.WebViewHandler
import me.tatarka.inject.annotations.Inject

@Inject
public class RealWebViewWalletCreator(
  private val webViewHandler: WebViewHandler,
  private val privyLogger: PrivyLogger,
): WalletCreator {
  public companion object {
    public const val diQualifierName: String = "WebViewWalletCreator"
  }

  override suspend fun createEthereumWallet(
    accessToken: String,
    existingSolanaWalletAddress: String?,
  ): Result<String> {
    return when (
      val createWalletResponse = webViewHandler.createEthereumWallet(
        accessToken = accessToken, existingSolanaWalletAddress = existingSolanaWalletAddress,
      )
    ) {
      is IFrameResponse.Error -> {
        // Some other error - surface it as failure
        val errorMessage = createWalletResponse.message
        privyLogger.error("Error creating Ethereum wallet: $errorMessage")
        Result.failure(EmbeddedWalletException(errorMessage))
      }
      is IFrameResponse.Success -> {
        val newWalletAddress = createWalletResponse.value.data.address
        privyLogger.debug("Newly created Ethereum wallet address: $newWalletAddress")

        Result.success(newWalletAddress)
      }
    }
  }

  override suspend fun createAdditionalEthereumWallet(
    accessToken: String,
    primaryWalletAddress: String,
    hdWalletIndex: Int,
  ): Result<String> {
    val response =
      webViewHandler.createAdditionalWallet(
        accessToken = accessToken,
        primaryWalletAddress = primaryWalletAddress,
        hdWalletIndex = hdWalletIndex,
      )

    return when (response) {
      is IFrameResponse.Error -> {
        // Some other error - surface it as failure
        val errorMessage = response.message
        privyLogger.error("Error creating additional wallet: $errorMessage")
        Result.failure(EmbeddedWalletException(errorMessage))
      }
      is IFrameResponse.Success -> {
        val newWalletAddress = response.value.data.address
        privyLogger.debug("Newly created wallet address: $newWalletAddress")
        Result.success(newWalletAddress)
      }
    }
  }

  override suspend fun createAdditionalSolanaWallet(
    accessToken: String,
    primaryWalletAddress: String,
    hdWalletIndex: Int,
  ): Result<String> {

    //TODO: IMPLEMENT PROPER LEGACY WALLET CREATION
    privyLogger.error("User already has a solana wallet.")

    return Result.failure(
      exception = EmbeddedWalletException(message = "User already has a solana wallet.")
    )

  }

  override suspend fun createSolanaWallet(
    accessToken: String,
    existingEthereumWalletAddress: String?,
  ): Result<String> {
    return when (val createWalletResponse =
      webViewHandler.createSolanaWallet(
        accessToken = accessToken,
        existingEthereumWalletAddress = existingEthereumWalletAddress)) {
      is IFrameResponse.Error -> {
        // Some other error - surface it as failure
        val errorMessage = createWalletResponse.message
        privyLogger.error("Error creating Solana wallet: $errorMessage")
        Result.failure(EmbeddedWalletException(errorMessage))
      }
      is IFrameResponse.Success -> {
        val newWalletAddress = createWalletResponse.value.data.publicKey
        privyLogger.debug("Newly created Solana wallet address: $newWalletAddress")

        Result.success(newWalletAddress)
      }
    }
  }
}