package io.privy.wallet.walletApi

import io.ktor.resources.Resource
import io.privy.auth.flatMap
import io.privy.di.PrivyAppId
import io.privy.network.AuthType
import io.privy.network.KtorWrapper
import io.privy.network.PrivyEnvironment
import io.privy.network.toResult
import io.privy.wallet.walletApi.create.WalletApiCreateRequest
import io.privy.wallet.walletApi.create.WalletApiCreateResponse
import io.privy.wallet.walletApi.rpc.WalletApiRpcRequest
import io.privy.wallet.walletApi.rpc.WalletApiRpcResponse
import io.privy.wallet.webview.UserSigner
import io.ktor.client.request.headers
import io.privy.network.baseUrl
import kotlinx.serialization.serializer
import me.tatarka.inject.annotations.Inject
import kotlin.io.encoding.Base64
import kotlin.io.encoding.ExperimentalEncodingApi

@Inject
public class RealWalletApiRepository(
    private val ktorWrapper: KtorWrapper,
    private val userSignerProvider: () -> UserSigner,
    private val appId: PrivyAppId,
    private val environment: PrivyEnvironment,
    private val jsonCanonicalizer: JsonCanonicalizer
) : WalletApiRepository {
    
    /**
     * Resource for wallet creation endpoint
     */
    @Resource("wallets")
    private object WalletCreateResource
    
    override suspend fun createWallet(
        request: WalletApiCreateRequest,
        token: String
    ): Result<WalletApiCreateResponse> {
        val authType = AuthType.Authorization(token)
        return ktorWrapper.postResult<WalletCreateResource, WalletApiCreateRequest, WalletApiCreateResponse>(
            resource = WalletCreateResource,
            body = request,
            authType = authType
        ).toResult()
    }
    
    /**
     * Resource for wallet RPC endpoint
     */
    @Resource("wallets/{walletId}/rpc")
    private data class WalletRpcResource(val walletId: String)
    
    @OptIn(ExperimentalEncodingApi::class)
    override suspend fun rpc(
        request: WalletApiRpcRequest,
        walletId: String,
        token: String
    ): Result<WalletApiRpcResponse> {
        // Create the resource with the walletId
        val resource = WalletRpcResource(walletId)
        
        // Construct the full URL for signing (needed for payload signature)
        val resourceUrl = "${environment.baseUrl()}wallets/${walletId}/rpc"
        
        // Create the payload
        val payload = WalletApiPayload(
            version = 1,
            url = resourceUrl,
            method = "POST",
            headers = mapOf("privy-app-id" to appId),
            body = request
        )
        val serializer = serializer<WalletApiPayload<WalletApiRpcRequest>>()


        // 1. Canonicalize the payload
        return jsonCanonicalizer.encode(serializer, payload).flatMap { message ->
            // 2. Sign the canonicalized payload
            userSignerProvider().signature(message)
        }.flatMap { signature ->
            // 3. Encode the signature and make the API call
            val signatureBase64 = Base64.encode(signature)
            
            // Use standard Authorization type and add signature header via builder
            ktorWrapper.postResult<WalletRpcResource, WalletApiRpcRequest, WalletApiRpcResponse>(
                resource = resource,
                body = request,
                authType = AuthType.Authorization(token),
                builder = {
                    // Add the signature header directly
                    headers {
                        append("privy-authorization-signature", signatureBase64)
                    }
                }
            ).toResult()
        }
    }
}