package io.privy.wallet.webview

import kotlinx.serialization.ExperimentalSerializationApi
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.JsonClassDiscriminator

/**
 * This class is meant to be used as the model to deserialize success responses received from the
 * webview after injecting javascript.
 *
 * In an ideal world, this is part of the impl module, and we map this class to another data class
 * that we expose publicly
 */
@OptIn(ExperimentalSerializationApi::class)
@Serializable
// Use "event" as discriminator for subclass type. Each subclass type should specify
// @SerialName annotation, whose value should be the expected "event" string.
@JsonClassDiscriminator("event")
public sealed interface PrivyIFrameSuccessResponse {
  @SerialName("id") public val requestId: String

  @Serializable
  @SerialName(IFrameReadyEventName)
  public data class IFrameReadyResponse(
      @SerialName("id") override val requestId: String,
  ) : PrivyIFrameSuccessResponse

  @Serializable
  @SerialName(CreateEthereumWalletEventName)
  public data class CreateEthereumWalletResponse(
      @SerialName("id") override val requestId: String,
      val data: CreateEthereumWalletResponseData
  ) : PrivyIFrameSuccessResponse

  @Serializable
  @SerialName(CreateSolanaWalletEventName)
  public data class CreateSolanaWalletResponse(
      @SerialName("id") override val requestId: String,
      val data: CreateSolanaWalletResponseData
  ) : PrivyIFrameSuccessResponse

  @Serializable
  @SerialName(CreateAdditionalWalletEventName)
  public data class CreateAdditionalWalletResponse(
      @SerialName("id") override val requestId: String,
      val data: CreateAdditionalWalletResponseData
  ) : PrivyIFrameSuccessResponse

  @Serializable
  @SerialName(WalletConnectEventName)
  public data class ConnectWalletResponse(
      @SerialName("id") override val requestId: String,
      val data: ConnectWalletResponseData
  ) : PrivyIFrameSuccessResponse

  @Serializable
  @SerialName(WalletRecoverEventName)
  public data class RecoverWalletResponse(
      @SerialName("id") override val requestId: String,
      val data: RecoverWalletResponseData
  ) : PrivyIFrameSuccessResponse

  @Serializable
  @SerialName(RpcRequestEventName)
  public data class WalletRpcResponse<T : Any>(
      @SerialName("id") override val requestId: String,
      val data: WalletRpcResponseData<T>
  ) : PrivyIFrameSuccessResponse
}

@Serializable public data class CreateEthereumWalletResponseData(val address: String)

@Serializable public data class CreateSolanaWalletResponseData(val publicKey: String)

@Serializable public data class CreateAdditionalWalletResponseData(val address: String)

@Serializable public data class ConnectWalletResponseData(val entropyId: String)

@Serializable public data class RecoverWalletResponseData(val entropyId: String)

@Serializable
public data class WalletRpcResponseData<T : Any>(val address: String, val response: Response<T>) {
  @Serializable public data class Response<T : Any>(val method: String, val data: T)
}

// Solana signMessage response data structure
@Serializable public data class SolanaSignMessageResponseData(val signature: String)
